(function( $ ) {
	'use strict';

	// Almanya şehirleri
	var almanyaSehirleri = [
		{id: 1, name: 'Berlin'},
		{id: 2, name: 'Münih'},
		{id: 3, name: 'Hamburg'},
		{id: 4, name: 'Köln'},
		{id: 5, name: 'Frankfurt'},
		{id: 6, name: 'Stuttgart'},
		{id: 7, name: 'Düsseldorf'},
		{id: 8, name: 'Dortmund'},
		{id: 9, name: 'Essen'},
		{id: 10, name: 'Leipzig'},
		{id: 11, name: 'Bremen'},
		{id: 12, name: 'Dresden'},
		{id: 13, name: 'Hannover'},
		{id: 14, name: 'Nürnberg'},
		{id: 15, name: 'Duisburg'},
		{id: 16, name: 'Bochum'},
		{id: 17, name: 'Wuppertal'},
		{id: 18, name: 'Bielefeld'},
		{id: 19, name: 'Bonn'},
		{id: 20, name: 'Münster'}
	];

	// KKTC şehirleri
	var kktcSehirleri = [
		{id: 1, name: 'Lefkoşa'},
		{id: 2, name: 'Girne'},
		{id: 3, name: 'Mağusa'},
		{id: 4, name: 'Güzelyurt'},
		{id: 5, name: 'İskele'},
		{id: 6, name: 'Lefke'},
		{id: 7, name: 'Dipkarpaz'},
		{id: 8, name: 'Yeni Boğaziçi'},
		{id: 9, name: 'Akdoğan'},
		{id: 10, name: 'Geçitkale'},
		{id: 11, name: 'Değirmenlik'},
		{id: 12, name: 'Karpaz'},
		{id: 13, name: 'Tatlısu'},
		{id: 14, name: 'Bafra'},
		{id: 15, name: 'Kumyalı'}
	];

	var IlIlceAPI = {
		cache: {
			provinces: null,
			districts: {}
		},
		
		// İlleri getir - Statik veri kullanılıyor
		getProvinces: function() {
			if (this.cache.provinces) {
				return Promise.resolve(this.cache.provinces);
			}
			
			var provinces = this.getFallbackProvinces();
			this.cache.provinces = provinces;
			return Promise.resolve(provinces);
		},
		
		// İlçeleri getir - Statik veri kullanılıyor
		getDistricts: function(provinceId) {
			// Almanya için özel işlem
			if (provinceId == -1) {
				return Promise.resolve(almanyaSehirleri);
			}
			
			// KKTC için özel işlem
			if (provinceId == -2) {
				return Promise.resolve(kktcSehirleri);
			}
			
			if (this.cache.districts[provinceId]) {
				return Promise.resolve(this.cache.districts[provinceId]);
			}
			
			var districts = this.getFallbackDistricts(provinceId);
			this.cache.districts[provinceId] = districts;
			return Promise.resolve(districts);
		},
		
		// Fallback: Eski statik veri
		getFallbackProvinces: function() {
			var fallbackData = [
				{il:"Almanya",plaka:-1},
				{il:"KKTC",plaka:-2},
				{il:"Adana",plaka:1},{il:"Adıyaman",plaka:2},{il:"Afyonkarahisar",plaka:3},
				{il:"Ağrı",plaka:4},{il:"Amasya",plaka:5},{il:"Ankara",plaka:6},
				{il:"Antalya",plaka:7},{il:"Artvin",plaka:8},{il:"Aydın",plaka:9},
				{il:"Balıkesir",plaka:10},{il:"Bilecik",plaka:11},{il:"Bingöl",plaka:12},
				{il:"Bitlis",plaka:13},{il:"Bolu",plaka:14},{il:"Burdur",plaka:15},
				{il:"Bursa",plaka:16},{il:"Çanakkale",plaka:17},{il:"Çankırı",plaka:18},
				{il:"Çorum",plaka:19},{il:"Denizli",plaka:20},{il:"Diyarbakır",plaka:21},
				{il:"Edirne",plaka:22},{il:"Elazığ",plaka:23},{il:"Erzincan",plaka:24},
				{il:"Erzurum",plaka:25},{il:"Eskişehir",plaka:26},{il:"Gaziantep",plaka:27},
				{il:"Giresun",plaka:28},{il:"Gümüşhane",plaka:29},{il:"Hakkari",plaka:30},
				{il:"Hatay",plaka:31},{il:"Isparta",plaka:32},{il:"Mersin",plaka:33},
				{il:"İstanbul",plaka:34},{il:"İzmir",plaka:35},{il:"Kars",plaka:36},
				{il:"Kastamonu",plaka:37},{il:"Kayseri",plaka:38},{il:"Kırklareli",plaka:39},
				{il:"Kırşehir",plaka:40},{il:"Kocaeli",plaka:41},{il:"Konya",plaka:42},
				{il:"Kütahya",plaka:43},{il:"Malatya",plaka:44},{il:"Manisa",plaka:45},
				{il:"Kahramanmaraş",plaka:46},{il:"Mardin",plaka:47},{il:"Muğla",plaka:48},
				{il:"Muş",plaka:49},{il:"Nevşehir",plaka:50},{il:"Niğde",plaka:51},
				{il:"Ordu",plaka:52},{il:"Rize",plaka:53},{il:"Sakarya",plaka:54},
				{il:"Samsun",plaka:55},{il:"Siirt",plaka:56},{il:"Sinop",plaka:57},
				{il:"Sivas",plaka:58},{il:"Tekirdağ",plaka:59},{il:"Tokat",plaka:60},
				{il:"Trabzon",plaka:61},{il:"Tunceli",plaka:62},{il:"Şanlıurfa",plaka:63},
				{il:"Uşak",plaka:64},{il:"Van",plaka:65},{il:"Yozgat",plaka:66},
				{il:"Zonguldak",plaka:67},{il:"Aksaray",plaka:68},{il:"Bayburt",plaka:69},
				{il:"Karaman",plaka:70},{il:"Kırıkkale",plaka:71},{il:"Batman",plaka:72},
				{il:"Şırnak",plaka:73},{il:"Bartın",plaka:74},{il:"Ardahan",plaka:75},
				{il:"Iğdır",plaka:76},{il:"Yalova",plaka:77},{il:"Karabük",plaka:78},
				{il:"Kilis",plaka:79},{il:"Osmaniye",plaka:80},{il:"Düzce",plaka:81}
			];
			
			return fallbackData.map(function(p) {
				return {id: p.plaka, name: p.il};
			});
		},
		
		getFallbackDistricts: function(provinceId) {
			// Eski statik veri - ilçeler
			var fallbackData = {
				1: ["Aladağ","Ceyhan","Çukurova","Feke","İmamoğlu","Karaisalı","Karataş","Kozan","Pozantı","Saimbeyli","Sarıçam","Seyhan","Tufanbeyli","Yumurtalık","Yüreğir"],
				2: ["Besni","Çelikhan","Gerger","Gölbaşı","Kahta","Merkez","Samsat","Sincik","Tut"],
				3: ["Başmakçı","Bayat","Bolvadin","Çay","Çobanlar","Dazkırı","Dinar","Emirdağ","Evciler","Hocalar","İhsaniye","İscehisar","Kızılören","Merkez","Sandıklı","Sinanpaşa","Sultandağı","Şuhut"],
				4: ["Diyadin","Doğubayazıt","Eleşkirt","Hamur","Merkez","Patnos","Taşlıçay","Tutak"],
				5: ["Göynücek","Gümüşhacıköy","Hamamözü","Merkez","Merzifon","Suluova","Taşova"],
				6: ["Altındağ","Ayaş","Bala","Beypazarı","Çamlıdere","Çankaya","Çubuk","Elmadağ","Güdül","Haymana","Kalecik","Kızılcahamam","Nallıhan","Polatlı","Şereflikoçhisar","Yenimahalle","Gölbaşı","Keçiören","Mamak","Sincan","Kazan","Akyurt","Etimesgut","Evren","Pursaklar"],
				7: ["Akseki","Alanya","Elmalı","Finike","Gazipaşa","Gündoğmuş","Kaş","Korkuteli","Kumluca","Manavgat","Serik","Demre","İbradı","Kemer","Aksu","Döşemealtı","Kepez","Konyaaltı","Muratpaşa"],
				8: ["Ardanuç","Arhavi","Merkez","Borçka","Hopa","Şavşat","Yusufeli","Murgul"],
				9: ["Merkez","Bozdoğan","Efeler","Çine","Germencik","Karacasu","Koçarlı","Kuşadası","Kuyucak","Nazilli","Söke","Sultanhisar","Yenipazar","Buharkent","İncirliova","Karpuzlu","Köşk","Didim"],
				10: ["Altıeylül","Ayvalık","Merkez","Balya","Bandırma","Bigadiç","Burhaniye","Dursunbey","Edremit","Erdek","Gönen","Havran","İvrindi","Karesi","Kepsut","Manyas","Savaştepe","Sındırgı","Gömeç","Susurluk","Marmara"],
				11: ["Merkez","Bozüyük","Gölpazarı","Osmaneli","Pazaryeri","Söğüt","Yenipazar","İnhisar"],
				12: ["Merkez","Genç","Karlıova","Kiğı","Solhan","Adaklı","Yayladere","Yedisu"],
				13: ["Adilcevaz","Ahlat","Merkez","Hizan","Mutki","Tatvan","Güroymak"],
				14: ["Merkez","Gerede","Göynük","Kıbrıscık","Mengen","Mudurnu","Seben","Dörtdivan","Yeniçağa"],
				15: ["Ağlasun","Bucak","Merkez","Gölhisar","Tefenni","Yeşilova","Karamanlı","Kemer","Altınyayla","Çavdır","Çeltikçi"],
				16: ["Gemlik","İnegöl","İznik","Karacabey","Keles","Mudanya","Mustafakemalpaşa","Orhaneli","Orhangazi","Yenişehir","Büyükorhan","Harmancık","Nilüfer","Osmangazi","Yıldırım","Gürsu","Kestel"],
				17: ["Ayvacık","Bayramiç","Biga","Bozcaada","Çan","Merkez","Eceabat","Ezine","Gelibolu","Gökçeada","Lapseki","Yenice"],
				18: ["Merkez","Çerkeş","Eldivan","Ilgaz","Kurşunlu","Orta","Şabanözü","Yapraklı","Atkaracalar","Kızılırmak","Bayramören","Korgun"],
				19: ["Alaca","Bayat","Merkez","İskilip","Kargı","Mecitözü","Ortaköy","Osmancık","Sungurlu","Boğazkale","Uğurludağ","Dodurga","Laçin","Oğuzlar"],
				20: ["Acıpayam","Buldan","Çal","Çameli","Çardak","Çivril","Merkez","Merkezefendi","Pamukkale","Güney","Kale","Sarayköy","Tavas","Babadağ","Bekilli","Honaz","Serinhisar","Baklan","Beyağaç","Bozkurt"],
				21: ["Kocaköy","Çermik","Çınar","Çüngüş","Dicle","Ergani","Hani","Hazro","Kulp","Lice","Silvan","Eğil","Bağlar","Kayapınar","Sur","Yenişehir","Bismil"],
				22: ["Merkez","Enez","Havsa","İpsala","Keşan","Lalapaşa","Meriç","Uzunköprü","Süloğlu"],
				23: ["Ağın","Baskil","Merkez","Karakoçan","Keban","Maden","Palu","Sivrice","Arıcak","Kovancılar","Alacakaya"],
				24: ["Çayırlı","Merkez","İliç","Kemah","Kemaliye","Refahiye","Tercan","Üzümlü","Otlukbeli"],
				25: ["Aşkale","Çat","Hınıs","Horasan","İspir","Karayazı","Narman","Oltu","Olur","Pasinler","Şenkaya","Tekman","Tortum","Karaçoban","Uzundere","Pazaryolu","Köprüköy","Palandöken","Yakutiye","Aziziye"],
				26: ["Çifteler","Mahmudiye","Mihalıççık","Sarıcakaya","Seyitgazi","Sivrihisar","Alpu","Beylikova","İnönü","Günyüzü","Han","Mihalgazi","Odunpazarı","Tepebaşı"],
				27: ["Araban","İslahiye","Nizip","Oğuzeli","Yavuzeli","Şahinbey","Şehitkamil","Karkamış","Nurdağı"],
				28: ["Alucra","Bulancak","Dereli","Espiye","Eynesil","Merkez","Görele","Keşap","Şebinkarahisar","Tirebolu","Piraziz","Yağlıdere","Çamoluk","Çanakçı","Doğankent","Güce"],
				29: ["Merkez","Kelkit","Şiran","Torul","Köse","Kürtün"],
				30: ["Çukurca","Merkez","Şemdinli","Yüksekova"],
				31: ["Altınözü","Arsuz","Defne","Dörtyol","Hassa","Antakya","İskenderun","Kırıkhan","Payas","Reyhanlı","Samandağ","Yayladağı","Erzin","Belen","Kumlu"],
				32: ["Atabey","Eğirdir","Gelendost","Merkez","Keçiborlu","Senirkent","Sütçüler","Şarkikaraağaç","Uluborlu","Yalvaç","Aksu","Gönen","Yenişarbademli"],
				33: ["Anamur","Erdemli","Gülnar","Mut","Silifke","Tarsus","Aydıncık","Bozyazı","Çamlıyayla","Akdeniz","Mezitli","Toroslar","Yenişehir"],
				34: ["Adalar","Bakırköy","Beşiktaş","Beykoz","Beyoğlu","Çatalca","Eyüp","Fatih","Gaziosmanpaşa","Kadıköy","Kartal","Sarıyer","Silivri","Şile","Şişli","Üsküdar","Zeytinburnu","Büyükçekmece","Kağıthane","Küçükçekmece","Pendik","Ümraniye","Bayrampaşa","Avcılar","Bağcılar","Bahçelievler","Güngören","Maltepe","Sultanbeyli","Tuzla","Esenler","Arnavutköy","Ataşehir","Başakşehir","Beylikdüzü","Çekmeköy","Esenyurt","Sancaktepe","Sultangazi"],
				35: ["Aliağa","Bayındır","Bergama","Bornova","Çeşme","Dikili","Foça","Karaburun","Karşıyaka","Kemalpaşa","Kınık","Kiraz","Menemen","Ödemiş","Seferihisar","Selçuk","Tire","Torbalı","Urla","Beydağ","Buca","Konak","Menderes","Balçova","Çiğli","Gaziemir","Narlıdere","Güzelbahçe","Bayraklı","Karabağlar"],
				36: ["Arpaçay","Digor","Kağızman","Merkez","Sarıkamış","Selim","Susuz","Akyaka"],
				37: ["Abana","Araç","Azdavay","Bozkurt","Cide","Çatalzeytin","Daday","Devrekani","İnebolu","Merkez","Küre","Taşköprü","Tosya","İhsangazi","Pınarbaşı","Şenpazar","Ağlı","Doğanyurt","Hanönü","Seydiler"],
				38: ["Bünyan","Develi","Felahiye","İncesu","Pınarbaşı","Sarıoğlan","Sarız","Tomarza","Yahyalı","Yeşilhisar","Akkışla","Talas","Kocasinan","Melikgazi","Hacılar","Özvatan"],
				39: ["Babaeski","Demirköy","Merkez","Kofçaz","Lüleburgaz","Pehlivanköy","Pınarhisar","Vize"],
				40: ["Çiçekdağı","Kaman","Merkez","Mucur","Akpınar","Akçakent","Boztepe"],
				41: ["Gebze","Gölcük","Kandıra","Karamürsel","Körfez","Derince","Başiskele","Çayırova","Darıca","Dilovası","İzmit","Kartepe"],
				42: ["Akşehir","Beyşehir","Bozkır","Cihanbeyli","Çumra","Doğanhisar","Ereğli","Hadim","Ilgın","Kadınhanı","Karapınar","Kulu","Sarayönü","Seydişehir","Yunak","Akören","Altınekin","Derebucak","Hüyük","Karatay","Meram","Selçuklu","Taşkent","Ahırlı","Çeltik","Derbent","Emirgazi","Güneysınır","Halkapınar","Tuzlukçu","Yalıhüyük"],
				43: ["Altıntaş","Domaniç","Emet","Gediz","Merkez","Simav","Tavşanlı","Aslanapa","Dumlupınar","Hisarcık","Şaphane","Çavdarhisar","Pazarlar"],
				44: ["Akçadağ","Arapgir","Arguvan","Darende","Doğanşehir","Hekimhan","Merkez","Pütürge","Yeşilyurt","Battalgazi","Doğanyol","Kale","Kuluncak","Yazıhan"],
				45: ["Akhisar","Alaşehir","Demirci","Gördes","Kırkağaç","Kula","Merkez","Salihli","Sarıgöl","Saruhanlı","Selendi","Soma","Şehzadeler","Yunusemre","Turgutlu","Ahmetli","Gölmarmara","Köprübaşı"],
				46: ["Afşin","Andırın","Dulkadiroğlu","Onikişubat","Elbistan","Göksun","Merkez","Pazarcık","Türkoğlu","Çağlayancerit","Ekinözü","Nurhak"],
				47: ["Derik","Kızıltepe","Artuklu","Merkez","Mazıdağı","Midyat","Nusaybin","Ömerli","Savur","Dargeçit","Yeşilli"],
				48: ["Bodrum","Datça","Fethiye","Köyceğiz","Marmaris","Menteşe","Milas","Ula","Yatağan","Dalaman","Seydikemer","Ortaca","Kavaklıdere"],
				49: ["Bulanık","Malazgirt","Merkez","Varto","Hasköy","Korkut"],
				50: ["Avanos","Derinkuyu","Gülşehir","Hacıbektaş","Kozaklı","Merkez","Ürgüp","Acıgöl"],
				51: ["Bor","Çamardı","Merkez","Ulukışla","Altunhisar","Çiftlik"],
				52: ["Akkuş","Altınordu","Aybastı","Fatsa","Gölköy","Korgan","Kumru","Mesudiye","Perşembe","Ulubey","Ünye","Gülyalı","Gürgentepe","Çamaş","Çatalpınar","Çaybaşı","İkizce","Kabadüz","Kabataş"],
				53: ["Ardeşen","Çamlıhemşin","Çayeli","Fındıklı","İkizdere","Kalkandere","Pazar","Merkez","Güneysu","Derepazarı","Hemşin","İyidere"],
				54: ["Akyazı","Geyve","Hendek","Karasu","Kaynarca","Sapanca","Kocaali","Pamukova","Taraklı","Ferizli","Karapürçek","Söğütlü","Adapazarı","Arifiye","Erenler","Serdivan"],
				55: ["Alaçam","Bafra","Çarşamba","Havza","Kavak","Ladik","Terme","Vezirköprü","Asarcık","Ondokuzmayıs","Salıpazarı","Tekkeköy","Ayvacık","Yakakent","Atakum","Canik","İlkadım"],
				56: ["Baykan","Eruh","Kurtalan","Pervari","Merkez","Şirvan","Tillo"],
				57: ["Ayancık","Boyabat","Durağan","Erfelek","Gerze","Merkez","Türkeli","Dikmen","Saraydüzü"],
				58: ["Divriği","Gemerek","Gürün","Hafik","İmranlı","Kangal","Koyulhisar","Merkez","Suşehri","Şarkışla","Yıldızeli","Zara","Akıncılar","Altınyayla","Doğanşar","Gölova","Ulaş"],
				59: ["Çerkezköy","Çorlu","Ergene","Hayrabolu","Malkara","Muratlı","Saray","Süleymanpaşa","Kapaklı","Şarköy","Marmaraereğlisi"],
				60: ["Almus","Artova","Erbaa","Niksar","Reşadiye","Merkez","Turhal","Zile","Pazar","Yeşilyurt","Başçiftlik","Sulusaray"],
				61: ["Akçaabat","Araklı","Arsin","Çaykara","Maçka","Of","Ortahisar","Sürmene","Tonya","Vakfıkebir","Yomra","Beşikdüzü","Şalpazarı","Çarşıbaşı","Dernekpazarı","Düzköy","Hayrat","Köprübaşı"],
				62: ["Çemişgezek","Hozat","Mazgirt","Nazımiye","Ovacık","Pertek","Pülümür","Merkez"],
				63: ["Akçakale","Birecik","Bozova","Ceylanpınar","Eyyübiye","Halfeti","Haliliye","Hilvan","Karaköprü","Siverek","Suruç","Viranşehir","Harran"],
				64: ["Banaz","Eşme","Karahallı","Sivaslı","Ulubey","Merkez"],
				65: ["Başkale","Çatak","Erciş","Gevaş","Gürpınar","İpekyolu","Muradiye","Özalp","Tuşba","Bahçesaray","Çaldıran","Edremit","Saray"],
				66: ["Akdağmadeni","Boğazlıyan","Çayıralan","Çekerek","Sarıkaya","Sorgun","Şefaatli","Yerköy","Merkez","Aydıncık","Çandır","Kadışehri","Saraykent","Yenifakılı"],
				67: ["Çaycuma","Devrek","Ereğli","Merkez","Alaplı","Gökçebey"],
				68: ["Ağaçören","Eskil","Gülağaç","Güzelyurt","Merkez","Ortaköy","Sarıyahşi"],
				69: ["Merkez","Aydıntepe","Demirözü"],
				70: ["Ermenek","Merkez","Ayrancı","Kazımkarabekir","Başyayla","Sarıveliler"],
				71: ["Delice","Keskin","Merkez","Sulakyurt","Bahşili","Balışeyh","Çelebi","Karakeçili","Yahşihan"],
				72: ["Merkez","Beşiri","Gercüş","Kozluk","Sason","Hasankeyf"],
				73: ["Beytüşşebap","Cizre","İdil","Silopi","Merkez","Uludere","Güçlükonak"],
				74: ["Merkez","Kurucaşile","Ulus","Amasra"],
				75: ["Merkez","Çıldır","Göle","Hanak","Posof","Damal"],
				76: ["Aralık","Merkez","Tuzluca","Karakoyunlu"],
				77: ["Merkez","Altınova","Armutlu","Çınarcık","Çiftlikköy","Termal"],
				78: ["Eflani","Eskipazar","Merkez","Ovacık","Safranbolu","Yenice"],
				79: ["Merkez","Elbeyli","Musabeyli","Polateli"],
				80: ["Bahçe","Kadirli","Merkez","Düziçi","Hasanbeyli","Sumbas","Toprakkale"],
				81: ["Akçakoca","Merkez","Yığılca","Cumayeri","Gölyaka","Çilimli","Gümüşova","Kaynaşlı"]
			};
			
			var districts = fallbackData[provinceId];
			if (districts && districts.length > 0) {
				return districts.map(function(d, index) {
					return {id: index + 1, name: d};
				});
			}
			return [];
		}
	};

	// Arama dropdown component
	var SearchDropdown = {
		init: function(inputId, dropdownId, selectId, onSelect) {
			var $input = $('#' + inputId);
			var $dropdown = $('#' + dropdownId);
			var $select = $('#' + selectId);
			var $clearBtn = $('#' + inputId.replace('-search', '-clear'));
			var items = [];
			var selectedItem = null;
			
			// Input focus ve click - tıklandığında veya focus olduğunda seçenekleri göster
			$input.on('focus click', function(e) {
				if (items.length > 0) {
					var searchTerm = $(this).val().toLowerCase();
					filterAndShow(searchTerm);
				}
			});
			
			// Input değişikliği - arama
			$input.on('input', function() {
				var searchTerm = $(this).val().toLowerCase();
				// Eğer seçili item varsa ve yazılan değer seçili item ile eşleşmiyorsa, seçimi temizle
				if (selectedItem && selectedItem.name.toLowerCase() !== searchTerm) {
					selectedItem = null;
					$select.val('');
				}
				filterAndShow(searchTerm);
				
				// Temizle butonunu göster/gizle
				if ($clearBtn.length) {
					if (searchTerm && selectedItem && selectedItem.name.toLowerCase() === searchTerm) {
						$clearBtn.show();
					} else if (!searchTerm) {
						$clearBtn.hide();
					}
				}
			});
			
			// Form submit'te manuel girişi engelle - sadece dropdown'dan seçilen değerler kabul edilsin
			$input.closest('form').on('submit', function(e) {
				// Eğer input'ta değer var ama dropdown'dan seçilmemişse, temizle
				if ($input.val() && !selectedItem) {
					$input.val('');
					if ($clearBtn.length) {
						$clearBtn.hide();
					}
				}
			});
			
			// Dışarı tıklanınca kapat
			$(document).on('click', function(e) {
				if (!$(e.target).closest('#' + inputId).length && 
				    !$(e.target).closest('#' + dropdownId).length) {
					$dropdown.hide();
				}
			});
			
			// Item click handler - event delegation kullan
			$dropdown.on('click', '.ilan-dropdown-item', function() {
				var id = $(this).data('id');
				var name = $(this).data('name');
				selectedItem = {id: id, name: name};
				$input.val(name);
				$select.val(id);
				
				// Hidden input'u da güncelle
				var hiddenInput = $('#' + selectId + '-hidden');
				if (hiddenInput.length) {
					hiddenInput.val(name);
				}
				
				// Temizle butonunu göster
				if ($clearBtn.length) {
					$clearBtn.show();
				}
				
				$dropdown.hide();
				
				if (onSelect) {
					onSelect(id, name);
				}
			});
			
			// Temizle butonu click
			if ($clearBtn.length) {
				$clearBtn.on('click', function(e) {
					e.preventDefault();
					e.stopPropagation();
					clearSelection();
				});
			}
			
			// Seçimi temizle fonksiyonu
			function clearSelection() {
				$input.val('');
				$select.val('');
				selectedItem = null;
				$dropdown.hide();
				
				// Hidden input'u da temizle
				var hiddenInput = $('#' + selectId + '-hidden');
				if (hiddenInput.length) {
					hiddenInput.val('');
				}
				
				// Temizle butonunu gizle
				if ($clearBtn.length) {
					$clearBtn.hide();
				}
				
				// Eğer il temizlendiyse, ilçeyi de temizle
				if (selectId === 'ilan-il') {
					// İlçe dropdown'unu temizle
					if (window.ilceDropdownInstance) {
						window.ilceDropdownInstance.clear();
						window.ilceDropdownInstance.disable();
					}
					
					// Placeholder'ları sıfırla
					$('#ilan-il-search').attr('placeholder', 'İl ara...');
					
					if (onSelect) {
						onSelect(null, null);
					}
				}
			}
			
			// Filtrele ve göster
			function filterAndShow(searchTerm) {
				var filtered = items.filter(function(item) {
					return item.name.toLowerCase().includes(searchTerm);
				});
				
				if (filtered.length === 0) {
					$dropdown.html('<div class="ilan-dropdown-item">Sonuç bulunamadı</div>');
				} else {
					var html = '';
					filtered.forEach(function(item) {
						html += '<div class="ilan-dropdown-item" data-id="' + item.id + '" data-name="' + 
						        item.name + '">' + item.name + '</div>';
					});
					$dropdown.html(html);
				}
				
				$dropdown.show();
			}
			
			// Public methods
			return {
				setItems: function(newItems) {
					items = newItems;
					if ($input.is(':focus') || $input.val()) {
						filterAndShow($input.val().toLowerCase());
					}
				},
				getSelected: function() {
					return selectedItem;
				},
				clear: function() {
					clearSelection();
				},
				disable: function() {
					$input.prop('disabled', true);
					$select.prop('disabled', true);
					$dropdown.hide();
				},
				enable: function() {
					$input.prop('disabled', false);
					$select.prop('disabled', false);
				},
				setValue: function(id, name) {
					$input.val(name);
					$select.val(id);
					selectedItem = {id: id, name: name};
					
					// Hidden input'u da güncelle
					var hiddenInput = $('#' + selectId + '-hidden');
					if (hiddenInput.length) {
						hiddenInput.val(name);
					}
					
					// Temizle butonunu göster
					if ($clearBtn.length && name) {
						$clearBtn.show();
					}
				},
				updatePlaceholder: function(text) {
					$input.attr('placeholder', text);
				}
			};
		}
	};

	// Widget formlarını initialize et
	function initWidgetForms() {
		$('.kanews-form-filter[data-widget-id]').each(function() {
			var $form = $(this);
			var widgetId = $form.data('widget-id');
			if(!widgetId) return;
			
			var $ilInput = $('#' + widgetId + '-ilan-il-search');
			var $ilceInput = $('#' + widgetId + '-ilan-ilce-search');
			
			if(!$ilInput.length) return;
			
			// İlçe dropdown initialize
			var ilceDropdown = null;
			if($ilceInput.length) {
				ilceDropdown = SearchDropdown.init(
					widgetId + '-ilan-ilce-search',
					widgetId + '-ilan-ilce-dropdown',
					widgetId + '-ilan-ilce'
				);
			}
			
			// İl dropdown initialize
			var ilDropdown = SearchDropdown.init(
				widgetId + '-ilan-il-search',
				widgetId + '-ilan-il-dropdown',
				widgetId + '-ilan-il',
				function(id, name) {
					if(ilceDropdown) {
						ilceDropdown.clear();
						ilceDropdown.disable();
					}
					
					if(id == -1 || id == -2) {
						$ilInput.attr('placeholder', 'Ülke ara...');
						$('#' + widgetId + '-ilan-il-hidden').attr('name', 'ulke');
						if(ilceDropdown && $ilceInput.length) {
							$ilceInput.attr('placeholder', 'Şehir ara...');
							$('#' + widgetId + '-ilan-ilce-hidden').attr('name', 'sehir');
							ilceDropdown.enable();
							IlIlceAPI.getDistricts(id).then(function(districts) {
								ilceDropdown.setItems(districts);
							});
						}
					} else if(id > 0) {
						$ilInput.attr('placeholder', 'İl ara...');
						$('#' + widgetId + '-ilan-il-hidden').attr('name', 'il');
						if(ilceDropdown && $ilceInput.length) {
							$ilceInput.attr('placeholder', 'İlçe ara...');
							$('#' + widgetId + '-ilan-ilce-hidden').attr('name', 'ilce');
							ilceDropdown.enable();
							IlIlceAPI.getDistricts(id).then(function(districts) {
								ilceDropdown.setItems(districts);
							});
						}
					}
				}
			);
			
			// İlleri yükle
			IlIlceAPI.getProvinces().then(function(provinces) {
				ilDropdown.setItems(provinces);
				$ilInput.attr('placeholder', 'İl ara...');
				
				// URL parametrelerini uygula
				function getUrlParameter(name) {
					var urlParams = new URLSearchParams(window.location.search);
					return urlParams.get(name);
				}
				
				var ilParam = getUrlParameter('il') || getUrlParameter('ulke');
				if(ilParam) {
					var foundProvince = provinces.find(function(p) {
						return p.name === ilParam || p.name.toLowerCase() === ilParam.toLowerCase();
					});
					
					if(foundProvince) {
						ilDropdown.setValue(foundProvince.id, foundProvince.name);
						
						var ilceParam = getUrlParameter('ilce') || getUrlParameter('sehir');
						if(ilceParam && ilceDropdown) {
							IlIlceAPI.getDistricts(foundProvince.id).then(function(districts) {
								ilceDropdown.setItems(districts);
								ilceDropdown.enable();
								
								if(foundProvince.id == -1 || foundProvince.id == -2) {
									$ilInput.attr('placeholder', 'Ülke ara...');
									$('#' + widgetId + '-ilan-il-hidden').attr('name', 'ulke');
									if($ilceInput.length) {
										$ilceInput.attr('placeholder', 'Şehir ara...');
										$('#' + widgetId + '-ilan-ilce-hidden').attr('name', 'sehir');
									}
								} else {
									$ilInput.attr('placeholder', 'İl ara...');
									$('#' + widgetId + '-ilan-il-hidden').attr('name', 'il');
									if($ilceInput.length) {
										$ilceInput.attr('placeholder', 'İlçe ara...');
										$('#' + widgetId + '-ilan-ilce-hidden').attr('name', 'ilce');
									}
								}
								
								var foundDistrict = districts.find(function(d) {
									return d.name === ilceParam || d.name.toLowerCase() === ilceParam.toLowerCase();
								});
								if(foundDistrict) {
									ilceDropdown.setValue(foundDistrict.id, foundDistrict.name);
								}
							});
						}
					}
				}
			});
			
			// Form submit
			$form.on('submit', function() {
				var pid = $form.find('input[name="p"]');
				if(pid.length && pid.val()) {
					$form.attr('action', window.location.origin + '/');
				}
				
				var ilSelected = ilDropdown.getSelected();
				var ilceSelected = ilceDropdown ? ilceDropdown.getSelected() : null;
				
				if(ilSelected && ilSelected.name) {
					$('#' + widgetId + '-ilan-il-hidden').val(ilSelected.name);
					if(ilSelected.id == -1 || ilSelected.id == -2) {
						$('#' + widgetId + '-ilan-il-hidden').attr('name', 'ulke');
					} else {
						$('#' + widgetId + '-ilan-il-hidden').attr('name', 'il');
					}
				} else {
					$('#' + widgetId + '-ilan-il-hidden').val('').attr('name', 'il');
				}
				
				if(ilceSelected && ilceSelected.name && ilceDropdown) {
					$('#' + widgetId + '-ilan-ilce-hidden').val(ilceSelected.name);
					if(ilSelected && (ilSelected.id == -1 || ilSelected.id == -2)) {
						$('#' + widgetId + '-ilan-ilce-hidden').attr('name', 'sehir');
					} else {
						$('#' + widgetId + '-ilan-ilce-hidden').attr('name', 'ilce');
					}
				} else if(ilceDropdown) {
					$('#' + widgetId + '-ilan-ilce-hidden').val('').attr('name', 'ilce');
				}
			});
		});
	}

	// Sidebar formlarını initialize et (genel fonksiyon)
	function initSidebarForm(ilSearchId, ilceSearchId, ilHiddenId, ilceHiddenId, ilDropdownId, ilceDropdownId, ilSelectId, ilceSelectId) {
		var $ilInput = $('#' + ilSearchId);
		var $ilceInput = $('#' + ilceSearchId);
		
		if(!$ilInput.length) return;
		
		var ilceDropdown = SearchDropdown.init(ilceSearchId, ilceDropdownId, ilceSelectId);
		
		var ilDropdown = SearchDropdown.init(ilSearchId, ilDropdownId, ilSelectId, function(id, name) {
			if(ilceDropdown) {
				ilceDropdown.clear();
				ilceDropdown.disable();
			}
			
			if(id == -1 || id == -2) {
				$ilInput.attr('placeholder', 'Ülke ara...');
				$('#' + ilHiddenId).attr('name', 'ulke');
				if(ilceDropdown && $ilceInput.length) {
					$ilceInput.attr('placeholder', 'Şehir ara...');
					$('#' + ilceHiddenId).attr('name', 'sehir');
					ilceDropdown.enable();
					IlIlceAPI.getDistricts(id).then(function(districts) {
						ilceDropdown.setItems(districts);
					});
				}
			} else if(id > 0) {
				$ilInput.attr('placeholder', 'İl ara...');
				$('#' + ilHiddenId).attr('name', 'il');
				if(ilceDropdown && $ilceInput.length) {
					$ilceInput.attr('placeholder', 'İlçe ara...');
					$('#' + ilceHiddenId).attr('name', 'ilce');
					ilceDropdown.enable();
					IlIlceAPI.getDistricts(id).then(function(districts) {
						ilceDropdown.setItems(districts);
					});
				}
			} else {
				$ilInput.attr('placeholder', 'İl ara...');
				$('#' + ilHiddenId).attr('name', 'il');
				$ilceInput.attr('placeholder', 'İlçe ara...');
				$('#' + ilceHiddenId).attr('name', 'ilce');
			}
		});
		
		// URL parametrelerini oku
		function getUrlParameter(name) {
			var urlParams = new URLSearchParams(window.location.search);
			return urlParams.get(name);
		}
		
		IlIlceAPI.getProvinces().then(function(provinces) {
			ilDropdown.setItems(provinces);
			$ilInput.attr('placeholder', 'İl ara...');
			
			var ilParam = getUrlParameter('il') || getUrlParameter('ulke');
			if(ilParam) {
				var foundProvince = provinces.find(function(p) {
					return p.name === ilParam || p.name.toLowerCase() === ilParam.toLowerCase();
				});
				
				if(foundProvince) {
					ilDropdown.setValue(foundProvince.id, foundProvince.name);
					
					var ilceParam = getUrlParameter('ilce') || getUrlParameter('sehir');
					if(ilceParam && ilceDropdown) {
						IlIlceAPI.getDistricts(foundProvince.id).then(function(districts) {
							ilceDropdown.setItems(districts);
							ilceDropdown.enable();
							
							if(foundProvince.id == -1 || foundProvince.id == -2) {
								$ilInput.attr('placeholder', 'Ülke ara...');
								$('#' + ilHiddenId).attr('name', 'ulke');
								if($ilceInput.length) {
									$ilceInput.attr('placeholder', 'Şehir ara...');
									$('#' + ilceHiddenId).attr('name', 'sehir');
								}
							} else {
								$ilInput.attr('placeholder', 'İl ara...');
								$('#' + ilHiddenId).attr('name', 'il');
								if($ilceInput.length) {
									$ilceInput.attr('placeholder', 'İlçe ara...');
									$('#' + ilceHiddenId).attr('name', 'ilce');
								}
							}
							
							var foundDistrict = districts.find(function(d) {
								return d.name === ilceParam || d.name.toLowerCase() === ilceParam.toLowerCase();
							});
							if(foundDistrict) {
								ilceDropdown.setValue(foundDistrict.id, foundDistrict.name);
							}
						});
					}
				}
			}
		});
		
		// Form submit
		$ilInput.closest('form').on('submit', function() {
			var ilSelected = ilDropdown.getSelected();
			var ilceSelected = ilceDropdown ? ilceDropdown.getSelected() : null;
			
			if(ilSelected && ilSelected.name) {
				$('#' + ilHiddenId).val(ilSelected.name);
				if(ilSelected.id == -1 || ilSelected.id == -2) {
					$('#' + ilHiddenId).attr('name', 'ulke');
				} else {
					$('#' + ilHiddenId).attr('name', 'il');
				}
			} else {
				$('#' + ilHiddenId).val('').attr('name', 'il');
			}
			
			if(ilceSelected && ilceSelected.name && ilceDropdown) {
				$('#' + ilceHiddenId).val(ilceSelected.name);
				if(ilSelected && (ilSelected.id == -1 || ilSelected.id == -2)) {
					$('#' + ilceHiddenId).attr('name', 'sehir');
				} else {
					$('#' + ilceHiddenId).attr('name', 'ilce');
				}
			} else if(ilceDropdown) {
				$('#' + ilceHiddenId).val('').attr('name', 'ilce');
			}
		});
	}

	// Ana fonksiyon - Sidebar formları için
	$(document).ready(function() {
		// İlan sidebar formu
		var $ilanIlInput = $('#ilan-il-search');
		if($ilanIlInput.length && !$ilanIlInput.closest('form').data('widget-id')) {
			initSidebarForm(
				'ilan-il-search',
				'ilan-ilce-search',
				'ilan-il-hidden',
				'ilan-ilce-hidden',
				'ilan-il-dropdown',
				'ilan-ilce-dropdown',
				'ilan-il',
				'ilan-ilce'
			);
		}
		
		// Firma sidebar formu
		var $firmaIlInput = $('#firma-il-search');
		if($firmaIlInput.length) {
			initSidebarForm(
				'firma-il-search',
				'firma-ilce-search',
				'firma-il-hidden',
				'firma-ilce-hidden',
				'firma-il-dropdown',
				'firma-ilce-dropdown',
				'firma-il',
				'firma-ilce'
			);
		}
		
		// Firma page header formu
		var $firmaHeaderIlInput = $('#firma-header-il-search');
		if($firmaHeaderIlInput.length) {
			initSidebarForm(
				'firma-header-il-search',
				'firma-header-ilce-search',
				'firma-header-il-hidden',
				'firma-header-ilce-hidden',
				'firma-header-il-dropdown',
				'firma-header-ilce-dropdown',
				'firma-header-il',
				'firma-header-ilce'
			);
		}
		
		// Widget formlarını initialize et
		initWidgetForms();
		
		// Dashboard formlarını initialize et
		initDashboardForms();
	});

	// Dashboard formlarını initialize et
	function initDashboardForms() {
		// İlan Ekle formu
		var $ilInputAdd = $('#dashboard-ilan-il-search');
		var $ilceInputAdd = $('#dashboard-ilan-ilce-search');
		
		if($ilInputAdd.length) {
			var ilceDropdownAdd = SearchDropdown.init(
				'dashboard-ilan-ilce-search',
				'dashboard-ilan-ilce-dropdown',
				'dashboard-ilan-ilce'
			);
			
			var ilDropdownAdd = SearchDropdown.init(
				'dashboard-ilan-il-search',
				'dashboard-ilan-il-dropdown',
				'dashboard-ilan-il',
				function(id, name) {
					if(ilceDropdownAdd) {
						ilceDropdownAdd.clear();
						ilceDropdownAdd.disable();
					}
					
					if(id == -1 || id == -2) {
						$ilInputAdd.attr('placeholder', 'Ülke ara...');
						$('#dashboard-ilan-il-hidden').attr('name', 'ulke');
						if(ilceDropdownAdd && $ilceInputAdd.length) {
							$ilceInputAdd.attr('placeholder', 'Şehir ara...');
							$('#dashboard-ilan-ilce-hidden').attr('name', 'sehir');
							ilceDropdownAdd.enable();
							IlIlceAPI.getDistricts(id).then(function(districts) {
								ilceDropdownAdd.setItems(districts);
							});
						}
					} else if(id > 0) {
						$ilInputAdd.attr('placeholder', 'İl ara...');
						$('#dashboard-ilan-il-hidden').attr('name', 'il');
						if(ilceDropdownAdd && $ilceInputAdd.length) {
							$ilceInputAdd.attr('placeholder', 'İlçe ara...');
							$('#dashboard-ilan-ilce-hidden').attr('name', 'ilce');
							ilceDropdownAdd.enable();
							IlIlceAPI.getDistricts(id).then(function(districts) {
								ilceDropdownAdd.setItems(districts);
							});
						}
					}
				}
			);
			
			IlIlceAPI.getProvinces().then(function(provinces) {
				ilDropdownAdd.setItems(provinces);
				$ilInputAdd.attr('placeholder', 'İl ara...');
			});
			
			// Form submit
			$ilInputAdd.closest('form').on('submit', function() {
				var ilSelected = ilDropdownAdd.getSelected();
				var ilceSelected = ilceDropdownAdd ? ilceDropdownAdd.getSelected() : null;
				
				if(ilSelected && ilSelected.name) {
					$('#dashboard-ilan-il-hidden').val(ilSelected.id > 0 ? ilSelected.id : ilSelected.name);
					if(ilSelected.id == -1 || ilSelected.id == -2) {
						$('#dashboard-ilan-il-hidden').attr('name', 'ulke').val(ilSelected.name);
					} else {
						$('#dashboard-ilan-il-hidden').attr('name', 'il');
					}
				} else {
					$('#dashboard-ilan-il-hidden').val('').attr('name', 'il');
				}
				
				if(ilceSelected && ilceSelected.name && ilceDropdownAdd) {
					$('#dashboard-ilan-ilce-hidden').val(ilceSelected.name);
					if(ilSelected && (ilSelected.id == -1 || ilSelected.id == -2)) {
						$('#dashboard-ilan-ilce-hidden').attr('name', 'sehir');
					} else {
						$('#dashboard-ilan-ilce-hidden').attr('name', 'ilce');
					}
				} else if(ilceDropdownAdd) {
					$('#dashboard-ilan-ilce-hidden').val('').attr('name', 'ilce');
				}
			});
		}
		
		// İlan Düzenle formu
		var $ilInputEdit = $('#dashboard-ilan-edit-il-search');
		var $ilceInputEdit = $('#dashboard-ilan-edit-ilce-search');
		
		if($ilInputEdit.length) {
			var ilceDropdownEdit = SearchDropdown.init(
				'dashboard-ilan-edit-ilce-search',
				'dashboard-ilan-edit-ilce-dropdown',
				'dashboard-ilan-edit-ilce'
			);
			
			var ilDropdownEdit = SearchDropdown.init(
				'dashboard-ilan-edit-il-search',
				'dashboard-ilan-edit-il-dropdown',
				'dashboard-ilan-edit-il',
				function(id, name) {
					if(ilceDropdownEdit) {
						ilceDropdownEdit.clear();
						ilceDropdownEdit.disable();
					}
					
					if(id == -1 || id == -2) {
						$ilInputEdit.attr('placeholder', 'Ülke ara...');
						$('#dashboard-ilan-edit-il-hidden').attr('name', 'ulke');
						if(ilceDropdownEdit && $ilceInputEdit.length) {
							$ilceInputEdit.attr('placeholder', 'Şehir ara...');
							$('#dashboard-ilan-edit-ilce-hidden').attr('name', 'sehir');
							ilceDropdownEdit.enable();
							IlIlceAPI.getDistricts(id).then(function(districts) {
								ilceDropdownEdit.setItems(districts);
							});
						}
					} else if(id > 0) {
						$ilInputEdit.attr('placeholder', 'İl ara...');
						$('#dashboard-ilan-edit-il-hidden').attr('name', 'il');
						if(ilceDropdownEdit && $ilceInputEdit.length) {
							$ilceInputEdit.attr('placeholder', 'İlçe ara...');
							$('#dashboard-ilan-edit-ilce-hidden').attr('name', 'ilce');
							ilceDropdownEdit.enable();
							IlIlceAPI.getDistricts(id).then(function(districts) {
								ilceDropdownEdit.setItems(districts);
							});
						}
					}
				}
			);
			
			IlIlceAPI.getProvinces().then(function(provinces) {
				ilDropdownEdit.setItems(provinces);
				$ilInputEdit.attr('placeholder', 'İl ara...');
				
				// Mevcut il değerini yükle - önce visible input'tan, sonra hidden input'tan
				var currentIlName = $ilInputEdit.val();
				var currentIlValue = $('#dashboard-ilan-edit-il-hidden').val();
				var foundProvince = null;
				
				if(currentIlName) {
					// Önce visible input'taki şehir ismi ile bul
					foundProvince = provinces.find(function(p) {
						return p.name === currentIlName || p.name.toLowerCase() === currentIlName.toLowerCase();
					});
				}
				
				if(!foundProvince && currentIlValue) {
					// Eğer bulunamadıysa, hidden input'taki plaka numarası ile bul
					var plakaNum = parseInt(currentIlValue);
					if(!isNaN(plakaNum) && plakaNum > 0) {
						foundProvince = provinces.find(function(p) {
							return p.id == plakaNum;
						});
					}
				}
				
				if(foundProvince) {
					ilDropdownEdit.setValue(foundProvince.id, foundProvince.name);
					
					// Placeholder ve name attribute'ları güncelle
					if(foundProvince.id == -1 || foundProvince.id == -2) {
						$ilInputEdit.attr('placeholder', 'Ülke ara...');
						$('#dashboard-ilan-edit-il-hidden').attr('name', 'ulke');
						if($ilceInputEdit.length) {
							$ilceInputEdit.attr('placeholder', 'Şehir ara...');
							$('#dashboard-ilan-edit-ilce-hidden').attr('name', 'sehir');
						}
					} else {
						$ilInputEdit.attr('placeholder', 'İl ara...');
						$('#dashboard-ilan-edit-il-hidden').attr('name', 'il');
						if($ilceInputEdit.length) {
							$ilceInputEdit.attr('placeholder', 'İlçe ara...');
							$('#dashboard-ilan-edit-ilce-hidden').attr('name', 'ilce');
						}
					}
					
					// İlçeleri yükle
					IlIlceAPI.getDistricts(foundProvince.id).then(function(districts) {
						ilceDropdownEdit.setItems(districts);
						ilceDropdownEdit.enable();
						
						// Mevcut ilçe değerini yükle
						var currentIlceName = $ilceInputEdit.val();
						var currentIlceValue = $('#dashboard-ilan-edit-ilce-hidden').val();
						
						if(currentIlceName || currentIlceValue) {
							var foundDistrict = districts.find(function(d) {
								return (currentIlceName && (d.name === currentIlceName || d.name.toLowerCase() === currentIlceName.toLowerCase())) ||
									   (currentIlceValue && (d.name === currentIlceValue || d.name.toLowerCase() === currentIlceValue.toLowerCase()));
							});
							if(foundDistrict) {
								ilceDropdownEdit.setValue(foundDistrict.id, foundDistrict.name);
							}
						}
					});
				}
			});
			
			// Form submit
			$ilInputEdit.closest('form').on('submit', function() {
				var ilSelected = ilDropdownEdit.getSelected();
				var ilceSelected = ilceDropdownEdit ? ilceDropdownEdit.getSelected() : null;
				
				if(ilSelected && ilSelected.name) {
					$('#dashboard-ilan-edit-il-hidden').val(ilSelected.id > 0 ? ilSelected.id : ilSelected.name);
					if(ilSelected.id == -1 || ilSelected.id == -2) {
						$('#dashboard-ilan-edit-il-hidden').attr('name', 'ulke').val(ilSelected.name);
					} else {
						$('#dashboard-ilan-edit-il-hidden').attr('name', 'il');
					}
				} else {
					$('#dashboard-ilan-edit-il-hidden').val('').attr('name', 'il');
				}
				
				if(ilceSelected && ilceSelected.name && ilceDropdownEdit) {
					$('#dashboard-ilan-edit-ilce-hidden').val(ilceSelected.name);
					if(ilSelected && (ilSelected.id == -1 || ilSelected.id == -2)) {
						$('#dashboard-ilan-edit-ilce-hidden').attr('name', 'sehir');
					} else {
						$('#dashboard-ilan-edit-ilce-hidden').attr('name', 'ilce');
					}
				} else if(ilceDropdownEdit) {
					$('#dashboard-ilan-edit-ilce-hidden').val('').attr('name', 'ilce');
				}
			});
		}
	}


})(jQuery);
