<?php
namespace Kanews;

class AutoDailyContent {
    
    private static $instance;
    private $post_type = 'auto_daily_content';
    private $meta_prefix = '_kanews_auto_daily_';
    
    public static function getInstance() {
        if ( null === static::$instance ) {
            static::$instance = new static();
        }
        return static::$instance;
    }
    
    private function __construct() {
        add_action('init', array($this, 'register_post_type'));
        add_action('add_meta_boxes', array($this, 'add_meta_boxes'));
        add_action('save_post_' . $this->post_type, array($this, 'save_meta_boxes'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Cron job
        add_action('kanews_daily_content_cron', array($this, 'process_daily_posts'));
        
        // İlk yüklemede cron job'ı başlat
        add_action('init', array($this, 'maybe_schedule_cron'));
        
        // Frontend'de içerikteki shortcode'ları işle (başlık zaten işlenmiş olarak kaydediliyor)
        add_filter('the_content', array($this, 'process_shortcodes_in_content'));
        
        // Gün sonunda içeriği işlenmiş haline kaydet
        add_action('kanews_save_processed_content', array($this, 'save_processed_content'));
        
        // Gün sonunda içerik kaydetme kontrolü
        add_action('wp', array($this, 'maybe_save_processed_content'));
        
        // Shortcode handler
        add_shortcode('kanews_daily_var', array($this, 'daily_var_shortcode'));
        
        // AJAX handlers
        add_action('wp_ajax_kanews_delete_daily_post', array($this, 'ajax_delete_daily_post'));
        add_action('wp_ajax_kanews_delete_daily_posts_bulk', array($this, 'ajax_delete_daily_posts_bulk'));
    }
    
    /**
     * Cron job'ı başlat (eğer yoksa)
     */
    public function maybe_schedule_cron() {
        if (!wp_next_scheduled('kanews_daily_content_cron')) {
            self::schedule_cron();
        }
    }
    
    /**
     * Post type kaydı
     */
    public function register_post_type() {
        $labels = array(
            'name'               => __('Otomatik Günlük İçerik', 'kanews'),
            'singular_name'      => __('Otomatik Günlük İçerik', 'kanews'),
            'menu_name'          => __('Günlük İçerik', 'kanews'),
            'add_new'            => __('Yeni Ekle', 'kanews'),
            'add_new_item'       => __('Yeni Günlük İçerik Ekle', 'kanews'),
            'edit_item'          => __('Günlük İçerik Düzenle', 'kanews'),
            'new_item'           => __('Yeni Günlük İçerik', 'kanews'),
            'view_item'          => __('Günlük İçerik Görüntüle', 'kanews'),
            'search_items'       => __('Günlük İçerik Ara', 'kanews'),
            'not_found'          => __('Bulunamadı', 'kanews'),
            'not_found_in_trash' => __('Çöpte bulunamadı', 'kanews'),
        );
        
        $args = array(
            'labels'              => $labels,
            'public'              => false, // Dışarıya görünür değil
            'publicly_queryable'  => false,
            'show_ui'              => true,
            'show_in_menu'         => true,
            'show_in_nav_menus'    => false,
            'show_in_admin_bar'    => true,
            'menu_position'        => 25,
            'menu_icon'            => 'dashicons-calendar-alt',
            'capability_type'      => 'post',
            'hierarchical'         => false,
            'supports'             => array('title', 'editor', 'thumbnail'),
            'has_archive'          => false,
            'rewrite'              => false,
            'query_var'            => false,
            'exclude_from_search'  => true,
        );
        
        register_post_type($this->post_type, $args);
    }
    
    /**
     * Meta box'ları ekle
     */
    public function add_meta_boxes() {
        add_meta_box(
            'kanews_auto_daily_settings',
            __('Günlük İçerik Ayarları', 'kanews'),
            array($this, 'render_settings_meta_box'),
            $this->post_type,
            'normal',
            'high'
        );
        
        add_meta_box(
            'kanews_auto_daily_variables',
            __('Kullanılabilir Değişkenler', 'kanews'),
            array($this, 'render_variables_meta_box'),
            $this->post_type,
            'side',
            'default'
        );
        
        add_meta_box(
            'kanews_auto_daily_posts',
            __('Bu Template ile Oluşturulan Yazılar', 'kanews'),
            array($this, 'render_posts_meta_box'),
            $this->post_type,
            'normal',
            'default'
        );
    }
    
    /**
     * Ayarlar meta box
     */
    public function render_settings_meta_box($post) {
        wp_nonce_field('kanews_auto_daily_save', 'kanews_auto_daily_nonce');
        
        $category = get_post_meta($post->ID, $this->meta_prefix . 'category', true);
        $publish_date = get_post_meta($post->ID, $this->meta_prefix . 'publish_date', true);
        $status = get_post_meta($post->ID, $this->meta_prefix . 'status', true);
        $status = $status ? $status : 'active';
        $region = get_post_meta($post->ID, $this->meta_prefix . 'region', true);
        $city = get_post_meta($post->ID, $this->meta_prefix . 'city', true);
        
        // Kategoriler
        $categories = get_categories(array('hide_empty' => false));
        
        // Türkiye şehirleri
        $turkey_cities = $this->get_turkey_cities();
        
        ?>
        <table class="form-table">
            <tr>
                <th><label for="kanews_auto_daily_category"><?php _e('Kategori', 'kanews'); ?></label></th>
                <td>
                    <select name="kanews_auto_daily_category" id="kanews_auto_daily_category" style="width: 100%;">
                        <option value=""><?php _e('Kategori Seç', 'kanews'); ?></option>
                        <?php foreach($categories as $cat): ?>
                            <option value="<?php echo esc_attr($cat->term_id); ?>" <?php selected($category, $cat->term_id); ?>>
                                <?php echo esc_html($cat->name); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </td>
            </tr>
            <tr>
                <th><label for="kanews_auto_daily_region"><?php _e('İl', 'kanews'); ?> *</label></th>
                <td>
                    <select name="kanews_auto_daily_region" id="kanews_auto_daily_region" style="width: 100%;">
                        <option value=""><?php _e('İl Seç', 'kanews'); ?></option>
                        <?php foreach($turkey_cities as $city_name): ?>
                            <option value="<?php echo esc_attr($city_name); ?>" <?php selected($region, $city_name); ?>>
                                <?php echo esc_html($city_name); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <p class="description"><?php _e('Hava durumu, eczane ve namaz vakitleri için il seçiniz.', 'kanews'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="kanews_auto_daily_city"><?php _e('İlçe', 'kanews'); ?></label></th>
                <td>
                    <input type="text" name="kanews_auto_daily_city" id="kanews_auto_daily_city" 
                           value="<?php echo esc_attr($city); ?>" style="width: 100%;" 
                           placeholder="<?php _e('İlçe adı (opsiyonel)', 'kanews'); ?>" />
                    <p class="description"><?php _e('Eczane ve namaz vakitleri için ilçe belirtebilirsiniz. Boş bırakılırsa il merkezi kullanılır.', 'kanews'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="kanews_auto_daily_publish_date"><?php _e('Yayın Tarihi', 'kanews'); ?></label></th>
                <td>
                    <input type="date" name="kanews_auto_daily_publish_date" id="kanews_auto_daily_publish_date" 
                           value="<?php echo esc_attr($publish_date); ?>" style="width: 100%;" />
                    <p class="description"><?php _e('Bu tarihte otomatik olarak post oluşturulacak. Boş bırakılırsa her gün oluşturulur.', 'kanews'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="kanews_auto_daily_status"><?php _e('Durum', 'kanews'); ?></label></th>
                <td>
                    <select name="kanews_auto_daily_status" id="kanews_auto_daily_status" style="width: 100%;">
                        <option value="active" <?php selected($status, 'active'); ?>><?php _e('Aktif', 'kanews'); ?></option>
                        <option value="inactive" <?php selected($status, 'inactive'); ?>><?php _e('Pasif', 'kanews'); ?></option>
                    </select>
                </td>
            </tr>
        </table>
        <?php
    }
    
    /**
     * Değişkenler meta box
     */
    public function render_variables_meta_box($post) {
        $region = get_post_meta($post->ID, $this->meta_prefix . 'region', true);
        $city = get_post_meta($post->ID, $this->meta_prefix . 'city', true);
        $location_name = $region ? $region : 'Seçilen İl';
        if ($city) {
            $location_name .= ' ' . $city;
        }
        
        $variables = array(
            '%%tarih%%' => __('Tarih (örn: 30 Eylül 2025)', 'kanews'),
            '%%il%%' => __('Seçilen İl', 'kanews'),
            '%%ilce%%' => __('Seçilen İlçe', 'kanews'),
            '%%hava%%' => sprintf(__('%s Hava Durumu', 'kanews'), $location_name),
            '%%eczane%%' => sprintf(__('%s Nöbetçi Eczaneler', 'kanews'), $location_name),
            '%%dolar%%' => __('Dolar Kuru (USD/TRY)', 'kanews'),
            '%%euro%%' => __('Euro Kuru (EUR/TRY)', 'kanews'),
            '%%bitcoin%%' => __('Bitcoin Fiyatı', 'kanews'),
            '%%ethereum%%' => __('Ethereum Fiyatı', 'kanews'),
            '%%altin%%' => __('Altın Fiyatı (Gram)', 'kanews'),
            '%%ceyrek_altin%%' => __('Çeyrek Altın Fiyatı', 'kanews'),
            '%%cumhuriyet_altin%%' => __('Cumhuriyet Altını Fiyatı', 'kanews'),
            '%%namaz%%' => sprintf(__('%s Namaz Vakitleri', 'kanews'), $location_name),
        );
        
        ?>
        <div class="kanews-variables-list">
            <p><?php _e('Başlık ve içerikte kullanabileceğiniz değişkenler:', 'kanews'); ?></p>
            <ul style="list-style: disc; padding-left: 20px;">
                <?php foreach($variables as $var => $desc): ?>
                    <li><code><?php echo esc_html($var); ?></code> - <?php echo esc_html($desc); ?></li>
                <?php endforeach; ?>
            </ul>
            <p style="margin-top: 10px; color: #666; font-size: 12px;">
                <?php _e('Not: Hava, eczane ve namaz değişkenleri seçtiğiniz il/ilçe bilgilerine göre çalışır.', 'kanews'); ?>
            </p>
        </div>
        <?php
    }
    
    /**
     * Meta box'ları kaydet
     */
    public function save_meta_boxes($post_id) {
        if (!isset($_POST['kanews_auto_daily_nonce']) || 
            !wp_verify_nonce($_POST['kanews_auto_daily_nonce'], 'kanews_auto_daily_save')) {
            return;
        }
        
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        if (isset($_POST['kanews_auto_daily_category'])) {
            update_post_meta($post_id, $this->meta_prefix . 'category', intval($_POST['kanews_auto_daily_category']));
        }
        
        if (isset($_POST['kanews_auto_daily_publish_date'])) {
            update_post_meta($post_id, $this->meta_prefix . 'publish_date', sanitize_text_field($_POST['kanews_auto_daily_publish_date']));
        }
        
        if (isset($_POST['kanews_auto_daily_status'])) {
            update_post_meta($post_id, $this->meta_prefix . 'status', sanitize_text_field($_POST['kanews_auto_daily_status']));
        }
        
        if (isset($_POST['kanews_auto_daily_region'])) {
            update_post_meta($post_id, $this->meta_prefix . 'region', sanitize_text_field($_POST['kanews_auto_daily_region']));
        }
        
        if (isset($_POST['kanews_auto_daily_city'])) {
            update_post_meta($post_id, $this->meta_prefix . 'city', sanitize_text_field($_POST['kanews_auto_daily_city']));
        }
    }
    
    /**
     * Template ile oluşturulan yazıları listele meta box
     */
    public function render_posts_meta_box($post) {
        // Bu template ile oluşturulan postları al
        $posts = get_posts(array(
            'post_type' => 'post',
            'posts_per_page' => -1,
            'meta_query' => array(
                array(
                    'key' => '_kanews_auto_daily_template_id',
                    'value' => $post->ID,
                    'compare' => '='
                )
            ),
            'orderby' => 'date',
            'order' => 'DESC'
        ));
        
        ?>
        <div class="kanews-template-posts-list">
            <?php if (empty($posts)): ?>
                <p><?php _e('Bu template ile henüz yazı oluşturulmamış.', 'kanews'); ?></p>
            <?php else: ?>
                <p style="margin-bottom: 15px;">
                    <strong><?php echo count($posts); ?></strong> <?php _e('yazı bulundu.', 'kanews'); ?>
                    <button type="button" class="button button-secondary" id="kanews-select-all-posts" style="margin-left: 10px;">
                        <?php _e('Tümünü Seç', 'kanews'); ?>
                    </button>
                    <button type="button" class="button button-secondary" id="kanews-deselect-all-posts" style="margin-left: 5px;">
                        <?php _e('Seçimi Kaldır', 'kanews'); ?>
                    </button>
                    <button type="button" class="button button-danger" id="kanews-delete-selected-posts" style="margin-left: 10px;" data-template-id="<?php echo esc_attr($post->ID); ?>">
                        <?php _e('Seçilenleri Sil', 'kanews'); ?>
                    </button>
                </p>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th style="width: 30px;">
                                <input type="checkbox" id="kanews-select-all-checkbox">
                            </th>
                            <th><?php _e('Başlık', 'kanews'); ?></th>
                            <th style="width: 120px;"><?php _e('Tarih', 'kanews'); ?></th>
                            <th style="width: 100px;"><?php _e('Durum', 'kanews'); ?></th>
                            <th style="width: 150px;"><?php _e('Veri Durumu', 'kanews'); ?></th>
                            <th style="width: 100px;"><?php _e('İşlemler', 'kanews'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($posts as $post_item): ?>
                            <tr data-post-id="<?php echo esc_attr($post_item->ID); ?>">
                                <td>
                                    <input type="checkbox" class="kanews-post-checkbox" value="<?php echo esc_attr($post_item->ID); ?>">
                                </td>
                                <td>
                                    <strong>
                                        <a href="<?php echo get_edit_post_link($post_item->ID); ?>" target="_blank">
                                            <?php echo esc_html($post_item->post_title); ?>
                                        </a>
                                    </strong>
                                </td>
                                <td>
                                    <?php echo get_the_date('d.m.Y H:i', $post_item->ID); ?>
                                </td>
                                <td>
                                    <?php 
                                    $status = $post_item->post_status;
                                    $status_labels = array(
                                        'publish' => __('Yayında', 'kanews'),
                                        'draft' => __('Taslak', 'kanews'),
                                        'pending' => __('İncelemede', 'kanews'),
                                        'trash' => __('Çöp', 'kanews')
                                    );
                                    echo isset($status_labels[$status]) ? $status_labels[$status] : $status;
                                    ?>
                                </td>
                                <td>
                                    <?php
                                    $content_processed = get_post_meta($post_item->ID, '_kanews_content_processed', true);
                                    $has_shortcode = has_shortcode($post_item->post_content, 'kanews_daily_var');
                                    
                                    if ($content_processed) {
                                        echo '<span style="color: #46b450; font-weight: bold;">✓ ' . __('İşlenmiş', 'kanews') . '</span>';
                                        echo '<br><small style="color: #666;">' . __('Gün sonunda kaydedildi', 'kanews') . '</small>';
                                    } else if ($has_shortcode) {
                                        echo '<span style="color: #f56e28; font-weight: bold;">⟳ ' . __('Otomatik Çekiliyor', 'kanews') . '</span>';
                                        echo '<br><small style="color: #666;">' . __('Her görüntülemede', 'kanews') . '</small>';
                                    } else {
                                        echo '<span style="color: #666;">' . __('Bilinmiyor', 'kanews') . '</span>';
                                    }
                                    ?>
                                </td>
                                <td>
                                    <button type="button" class="button button-small kanews-delete-single-post" 
                                            data-post-id="<?php echo esc_attr($post_item->ID); ?>"
                                            data-template-id="<?php echo esc_attr($post->ID); ?>">
                                        <?php _e('Sil', 'kanews'); ?>
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        
        <style>
            .kanews-template-posts-list {
                margin-top: 10px;
            }
            .kanews-template-posts-list table {
                margin-top: 10px;
            }
            .kanews-template-posts-list .button-danger {
                background: #dc3232;
                border-color: #dc3232;
                color: #fff;
            }
            .kanews-template-posts-list .button-danger:hover {
                background: #a00;
                border-color: #a00;
            }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            // Tümünü seç
            $('#kanews-select-all-checkbox, #kanews-select-all-posts').on('click', function() {
                $('.kanews-post-checkbox').prop('checked', true);
            });
            
            // Seçimi kaldır
            $('#kanews-deselect-all-posts').on('click', function() {
                $('.kanews-post-checkbox').prop('checked', false);
            });
            
            // Tekil silme
            $('.kanews-delete-single-post').on('click', function() {
                if (!confirm('<?php echo esc_js(__('Bu yazıyı silmek istediğinize emin misiniz?', 'kanews')); ?>')) {
                    return;
                }
                
                var $button = $(this);
                var postId = $button.data('post-id');
                var templateId = $button.data('template-id');
                
                $button.prop('disabled', true).text('<?php echo esc_js(__('Siliniyor...', 'kanews')); ?>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'kanews_delete_daily_post',
                        post_id: postId,
                        template_id: templateId,
                        nonce: '<?php echo wp_create_nonce('kanews_delete_post'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            $button.closest('tr').fadeOut(300, function() {
                                $(this).remove();
                                // Eğer tablo boşaldıysa mesaj göster
                                if ($('.kanews-post-checkbox').length === 0) {
                                    $('.kanews-template-posts-list').html('<p><?php echo esc_js(__('Bu template ile henüz yazı oluşturulmamış.', 'kanews')); ?></p>');
                                }
                            });
                        } else {
                            alert(response.data.message || '<?php echo esc_js(__('Silme işlemi başarısız oldu.', 'kanews')); ?>');
                            $button.prop('disabled', false).text('<?php echo esc_js(__('Sil', 'kanews')); ?>');
                        }
                    },
                    error: function() {
                        alert('<?php echo esc_js(__('Bir hata oluştu.', 'kanews')); ?>');
                        $button.prop('disabled', false).text('<?php echo esc_js(__('Sil', 'kanews')); ?>');
                    }
                });
            });
            
            // Toplu silme
            $('#kanews-delete-selected-posts').on('click', function() {
                var selected = $('.kanews-post-checkbox:checked');
                
                if (selected.length === 0) {
                    alert('<?php echo esc_js(__('Lütfen silmek için en az bir yazı seçin.', 'kanews')); ?>');
                    return;
                }
                
                if (!confirm('<?php echo esc_js(__('Seçili', 'kanews')); ?> ' + selected.length + ' <?php echo esc_js(__('yazıyı silmek istediğinize emin misiniz?', 'kanews')); ?>')) {
                    return;
                }
                
                var $button = $(this);
                var templateId = $button.data('template-id');
                var postIds = [];
                
                selected.each(function() {
                    postIds.push($(this).val());
                });
                
                $button.prop('disabled', true).text('<?php echo esc_js(__('Siliniyor...', 'kanews')); ?>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'kanews_delete_daily_posts_bulk',
                        post_ids: postIds,
                        template_id: templateId,
                        nonce: '<?php echo wp_create_nonce('kanews_delete_post'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            selected.closest('tr').fadeOut(300, function() {
                                $(this).remove();
                                // Eğer tablo boşaldıysa mesaj göster
                                if ($('.kanews-post-checkbox').length === 0) {
                                    $('.kanews-template-posts-list').html('<p><?php echo esc_js(__('Bu template ile henüz yazı oluşturulmamış.', 'kanews')); ?></p>');
                                }
                            });
                            $button.prop('disabled', false).text('<?php echo esc_js(__('Seçilenleri Sil', 'kanews')); ?>');
                        } else {
                            alert(response.data.message || '<?php echo esc_js(__('Silme işlemi başarısız oldu.', 'kanews')); ?>');
                            $button.prop('disabled', false).text('<?php echo esc_js(__('Seçilenleri Sil', 'kanews')); ?>');
                        }
                    },
                    error: function() {
                        alert('<?php echo esc_js(__('Bir hata oluştu.', 'kanews')); ?>');
                        $button.prop('disabled', false).text('<?php echo esc_js(__('Seçilenleri Sil', 'kanews')); ?>');
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * Türkiye şehirleri listesi
     */
    private function get_turkey_cities() {
        $cities = array(
            'Adana', 'Adıyaman', 'Afyonkarahisar', 'Ağrı', 'Aksaray', 'Amasya', 'Ankara', 'Antalya',
            'Ardahan', 'Artvin', 'Aydın', 'Balıkesir', 'Bartın', 'Batman', 'Bayburt', 'Bilecik',
            'Bingöl', 'Bitlis', 'Bolu', 'Burdur', 'Bursa', 'Çanakkale', 'Çankırı', 'Çorum',
            'Denizli', 'Diyarbakır', 'Düzce', 'Edirne', 'Elazığ', 'Erzincan', 'Erzurum', 'Eskişehir',
            'Gaziantep', 'Giresun', 'Gümüşhane', 'Hakkâri', 'Hatay', 'Iğdır', 'Isparta', 'İstanbul',
            'İzmir', 'Kahramanmaraş', 'Karabük', 'Karaman', 'Kars', 'Kastamonu', 'Kayseri', 'Kırıkkale',
            'Kırklareli', 'Kırşehir', 'Kilis', 'Kocaeli', 'Konya', 'Kütahya', 'Malatya', 'Manisa',
            'Mardin', 'Mersin', 'Muğla', 'Muş', 'Nevşehir', 'Niğde', 'Ordu', 'Osmaniye', 'Rize',
            'Sakarya', 'Samsun', 'Siirt', 'Sinop', 'Sivas', 'Şırnak', 'Tekirdağ', 'Tokat',
            'Trabzon', 'Tunceli', 'Şanlıurfa', 'Uşak', 'Van', 'Yalova', 'Yozgat', 'Zonguldak'
        );
        
        sort($cities);
        return $cities;
    }
    
    /**
     * Admin script'leri
     */
    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'post.php' && $hook !== 'post-new.php') {
            return;
        }
        
        global $post_type;
        if ($post_type !== $this->post_type) {
            return;
        }
        
        // jQuery ve ajaxurl zaten WordPress admin'de mevcut
    }
    
    /**
     * Hava durumu verisi
     */
    private function get_weather_data($city) {
        // API'den çek
        $apikey = kanews_get_option('openweathermap-api');
        if (!$apikey) {
            return '';
        }
        
        
        
        $api_url = 'http://api.openweathermap.org/data/2.5/weather?q=' . urlencode($city) . '&lang=tr&units=metric&appid=' . $apikey;
        $response = wp_remote_get($api_url, array('timeout' => 10));
        
        if (is_wp_error($response)) {
            return '';
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (isset($data['main']['temp']) && isset($data['weather'][0]['description'])) {
            $temp = round($data['main']['temp']);
            $desc = ucfirst($data['weather'][0]['description']);
            return sprintf('%s°C, %s', $temp, $desc);
        }
        
        return '';
    }
    
    /**
     * Eczane verisi
     */
    private function get_pharmacy_data($region, $city) {
        if (!function_exists('kanews_eczane_times_api')) {
            return '';
        }
        
        $data = kanews_eczane_times_api($region, $city);
        
        if (isset($data['data'][0]) && is_array($data['data'][0]) && !empty($data['data'][0])) {
            $pharmacies = array_slice($data['data'][0], 0, 3); // İlk 3 eczane
            $output = array();
            foreach ($pharmacies as $pharmacy) {
                if (isset($pharmacy['eczane_adi']) && isset($pharmacy['adres'])) {
                    $output[] = $pharmacy['eczane_adi'] . ' (' . $pharmacy['adres'] . ')';
                }
            }
            return !empty($output) ? implode(', ', $output) : '';
        }
        
        return '';
    }
    
    /**
     * Döviz verisi
     */
    private function get_currency_data($currency = 'USD') {
        if (!function_exists('get_local_file_contents')) {
            return '';
        }
        
        
        
        $response = get_local_file_contents(KAN_TEMPLATE_PATH . "/inc/api/currencies/currencies.json");
        $data = json_decode($response, true);
        
        if (isset($data['data']) && isset($data['data'][$currency]['Selling'])) {
            return number_format((float)$data['data'][$currency]['Selling'], 2, ',', '.') . ' ₺';
        }
        
        return '';
    }
    
    /**
     * Altın verisi (Gram Altın)
     */
    private function get_gold_data() {
        if (!function_exists('get_local_file_contents')) {
            return '';
        }
        
        $response = get_local_file_contents(KAN_TEMPLATE_PATH . "/inc/api/gold/gold.json");
        $data = json_decode($response, true);

        if( isset($data['data']) && isset($data['data']['GRA']['Selling']) ) {
            return number_format((float)$data['data']['GRA']['Selling'], 2, ',', '.') . ' ₺';
        }
        
        return '';
    }
    
    /**
     * Çeyrek Altın verisi
     */
    private function get_quarter_gold_data() {
        if (!function_exists('get_local_file_contents')) {
            return '';
        }
        
        $response = get_local_file_contents(KAN_TEMPLATE_PATH . "/inc/api/gold/gold.json");
        $data = json_decode($response, true);
        
        if (isset($data['data']) && isset($data['data']['CEYREKALTIN']['Selling'])) {
            return number_format((float)$data['data']['CEYREKALTIN']['Selling'], 2, ',', '.') . ' ₺';
        }
        
        return '';
    }
    
    /**
     * Cumhuriyet Altını verisi
     */
    private function get_republic_gold_data() {
        if (!function_exists('get_local_file_contents')) {
            return '';
        }
        
        $response = get_local_file_contents(KAN_TEMPLATE_PATH . "/inc/api/gold/gold.json");
        $data = json_decode($response, true);
        
        if (isset($data['data']) && isset($data['data']['CUMHURIYETALTINI']['Selling'])) {
            return number_format((float)$data['data']['CUMHURIYETALTINI']['Selling'], 2, ',', '.') . ' ₺';
        }
        
        return '';
    }
    
    /**
     * Bitcoin verisi
     */
    private function get_bitcoin_data() {
        $index = kanews_get_option('crypto-page-index', 'try');
        
        if (!function_exists('kanews_crypto_data')) {
            return '';
        }
        
        $data = json_decode(kanews_crypto_data($index));
        
        if (!empty($data)) {
            foreach ($data as $coin) {
                if (strtolower($coin->id) === 'bitcoin' || strtolower($coin->symbol) === 'btc') {
                    return number_format((float)$coin->current_price, 2, ',', '.') . ' ' . strtoupper($index);
                }
            }
        }
        
        return '';
    }
    
    /**
     * Ethereum verisi
     */
    private function get_ethereum_data() {
        $index = kanews_get_option('crypto-page-index', 'try');
        
        if (!function_exists('kanews_crypto_data')) {
            return '';
        }
        
        $data = json_decode(kanews_crypto_data($index));
        
        if (!empty($data)) {
            foreach ($data as $coin) {
                if (strtolower($coin->id) === 'ethereum' || strtolower($coin->symbol) === 'eth') {
                    return number_format((float)$coin->current_price, 2, ',', '.') . ' ' . strtoupper($index);
                }
            }
        }
        
        return '';
    }
    
    /**
     * Namaz vakitleri verisi
     */
    private function get_prayer_data($region, $city) {
        if (!function_exists('kanews_prayer_times_api')) {
            return '';
        }
        
        $date = date('Y-m-d');
        $data = kanews_prayer_times_api('Turkey', $region, $city, $date, 1);
        
        if (isset($data['times']) && isset($data['times'][$date]) && is_array($data['times'][$date])) {
            $times = $data['times'][$date];
            if (count($times) >= 6) {
                return sprintf(
                    'İmsak: %s, Güneş: %s, Öğle: %s, İkindi: %s, Akşam: %s, Yatsı: %s',
                    $times[0], $times[1], $times[2], $times[3], $times[4], $times[5]
                );
            }
        }
        
        return '';
    }
    
    /**
     * Günlük post'ları işle
     */
    public function process_daily_posts() {
        $today = date('Y-m-d');
        
        // Aktif template'leri al
        $templates = get_posts(array(
            'post_type' => $this->post_type,
            'posts_per_page' => -1,
            'meta_query' => array(
                array(
                    'key' => $this->meta_prefix . 'status',
                    'value' => 'active',
                    'compare' => '='
                )
            )
        ));
        
        foreach ($templates as $template) {
            $publish_date = get_post_meta($template->ID, $this->meta_prefix . 'publish_date', true);
            
            // Eğer belirli bir tarih seçilmişse, sadece o tarihte oluştur
            if (!empty($publish_date) && $publish_date !== $today) {
                continue;
            }
            
            // Bugün için zaten post oluşturulmuş mu kontrol et
            // $existing_post = $this->get_existing_post($template->ID, $today);
            // if ($existing_post) {
            //    continue;
            // }
            
            // Post oluştur
            $this->create_daily_post($template, $today);
        }
    }
    
    /**
     * Mevcut post'u kontrol et
     */
    private function get_existing_post($template_id, $date) {
        $posts = get_posts(array(
            'post_type' => 'post',
            'posts_per_page' => 1,
            'meta_query' => array(
                array(
                    'key' => '_kanews_auto_daily_template_id',
                    'value' => $template_id,
                    'compare' => '='
                ),
                array(
                    'key' => '_kanews_auto_daily_date',
                    'value' => $date,
                    'compare' => '='
                )
            )
        ));
        
        return !empty($posts) ? $posts[0] : false;
    }
    
    /**
     * Günlük post oluştur
     */
    private function create_daily_post($template, $date) {
        $region = get_post_meta($template->ID, $this->meta_prefix . 'region', true);
        $city = get_post_meta($template->ID, $this->meta_prefix . 'city', true);
        
        if (!$region) {
            return false; // İl seçilmemişse post oluşturma
        }
        
        // Verileri direkt çek (option'da tutma)
        $data = array(
            'date' => $date,
            'template_id' => $template->ID,
            'region' => $region,
            'city' => $city,
            'hava' => $this->get_weather_data($region),
            'eczane' => $this->get_pharmacy_data($region, $city),
            'dolar' => $this->get_currency_data('USD'),
            'euro' => $this->get_currency_data('EUR'),
            'bitcoin' => $this->get_bitcoin_data(),
            'ethereum' => $this->get_ethereum_data(),
            'altin' => $this->get_gold_data(),
            'ceyrek_altin' => $this->get_quarter_gold_data(),
            'cumhuriyet_altin' => $this->get_republic_gold_data(),
            'namaz' => $this->get_prayer_data($region, $city ? $city : $region),
        );
        
        // Başlığı direkt işle (shortcode olmadan)
        $title = $this->replace_variables($template->post_title, $date, $data);
        
        // İçeriği shortcode formatına çevir (frontend'de işlenecek)
        $content = $this->convert_variables_to_shortcode($template->post_content);
        
        // Slug için başlığı kullan (zaten işlenmiş)
        $post_name = sanitize_title($title);
        
        // Post oluştur
        $post_data = array(
            'post_title' => $title,
            'post_content' => $content,
            'post_name' => $post_name,
            'post_status' => 'publish',
            'post_type' => 'post',
            'post_author' => 1,
            'post_date' => $date . ' 00:00:00',
        );
        
        $category = get_post_meta($template->ID, $this->meta_prefix . 'category', true);
        if ($category) {
            $post_data['post_category'] = array($category);
        }
        
        $post_id = wp_insert_post($post_data);
        
        if ($post_id && !is_wp_error($post_id)) {
            // Meta data ekle
            if( !empty($template->ID) ) 
            update_post_meta($post_id, '_kanews_auto_daily_template_id', $template->ID);
            if( !empty($date) ) 
            update_post_meta($post_id, '_kanews_auto_daily_date', $date);
            if( !empty($region) ) 
            update_post_meta($post_id, '_kanews_auto_daily', true);
            if( !empty($region) ) 
            update_post_meta($post_id, '_kanews_auto_daily_region', $region);
            if( !empty($city) ) 
            update_post_meta($post_id, '_kanews_auto_daily_city', $city);
            
            // Featured image'ı template'den al ve yeni posta ata
            $template_thumbnail_id = get_post_thumbnail_id($template->ID);
            if ($template_thumbnail_id) {
                // Öne çıkarılan görsel olarak ata
                update_post_meta($post_id, '_thumbnail_id', $template_thumbnail_id);
                set_post_thumbnail($post_id, $template_thumbnail_id);
            }
        }
        
        return $post_id;
    }
    
    /**
     * Slug için shortcode'ları işle
     */
    private function process_shortcode_for_slug($text, $date, $data) {
        // Shortcode pattern'ini bul
        $pattern = '/\[kanews_daily_var var="([^"]+)"\]/';
        
        return preg_replace_callback($pattern, function($matches) use ($date, $data) {
            $var_name = $matches[1];
            
            // Değişken adını data key'e çevir
            $var_map = array(
                'tarih' => 'date',
                'il' => 'region',
                'ilce' => 'city',
                'hava' => 'hava',
                'eczane' => 'eczane',
                'dolar' => 'dolar',
                'euro' => 'euro',
                'bitcoin' => 'bitcoin',
                'ethereum' => 'ethereum',
                'altin' => 'altin',
                'ceyrek_altin' => 'ceyrek_altin',
                'cumhuriyet_altin' => 'cumhuriyet_altin',
                'namaz' => 'namaz',
            );
            
            $data_field = isset($var_map[$var_name]) ? $var_map[$var_name] : $var_name;
            
            if ($var_name === 'tarih') {
                // Tarih formatı
                $date_obj = new \DateTime($date);
                $formatted_date = $date_obj->format('d F Y');
                $formatted_date = str_replace(
                    array('January', 'February', 'March', 'April', 'May', 'June', 
                          'July', 'August', 'September', 'October', 'November', 'December'),
                    array('Ocak', 'Şubat', 'Mart', 'Nisan', 'Mayıs', 'Haziran',
                          'Temmuz', 'Ağustos', 'Eylül', 'Ekim', 'Kasım', 'Aralık'),
                    $formatted_date
                );
                return $formatted_date;
            }
            
            return isset($data[$data_field]) ? $data[$data_field] : '';
        }, $text);
    }
    
    /**
     * Değişkenleri shortcode formatına çevir
     */
    private function convert_variables_to_shortcode($text) {
        $variable_map = array(
            '%%tarih%%' => 'tarih',
            '%%il%%' => 'il',
            '%%ilce%%' => 'ilce',
            '%%hava%%' => 'hava',
            '%%eczane%%' => 'eczane',
            '%%dolar%%' => 'dolar',
            '%%euro%%' => 'euro',
            '%%bitcoin%%' => 'bitcoin',
            '%%ethereum%%' => 'ethereum',
            '%%altin%%' => 'altin',
            '%%ceyrek_altin%%' => 'ceyrek_altin',
            '%%cumhuriyet_altin%%' => 'cumhuriyet_altin',
            '%%namaz%%' => 'namaz',
        );
        
        foreach ($variable_map as $variable => $var_name) {
            $text = str_replace($variable, '[kanews_daily_var var="' . $var_name . '"]', $text);
        }
        
        return $text;
    }
    
    /**
     * Günlük içerik değişkenleri shortcode handler
     */
    public function daily_var_shortcode($atts) {
        global $post;
        
        if (!$post) {
            return '';
        }
        
        // Sadece otomatik oluşturulan postlar için
        if (!get_post_meta($post->ID, '_kanews_auto_daily', true)) {
            return '';
        }
        
        $atts = shortcode_atts(array(
            'var' => ''
        ), $atts);
        
        if (empty($atts['var'])) {
            return '';
        }
        
        $date = get_post_meta($post->ID, '_kanews_auto_daily_date', true);
        if (!$date) {
            $date = date('Y-m-d', strtotime(get_the_date('Y-m-d', $post->ID)));
        }
        
        $template_id = get_post_meta($post->ID, '_kanews_auto_daily_template_id', true);
        if (!$template_id) {
            return '';
        }
        
        // Sadece kullanılan shortcode için gerekli veriyi çek
        $var_name = $atts['var'];
        
        // Tarih için
        if ($var_name === 'tarih') {
            $date_obj = new \DateTime($date);
            $formatted_date = $date_obj->format('d F Y');
            $formatted_date = str_replace(
                array('January', 'February', 'March', 'April', 'May', 'June', 
                      'July', 'August', 'September', 'October', 'November', 'December'),
                array('Ocak', 'Şubat', 'Mart', 'Nisan', 'Mayıs', 'Haziran',
                      'Temmuz', 'Ağustos', 'Eylül', 'Ekim', 'Kasım', 'Aralık'),
                $formatted_date
            );
            return $formatted_date;
        }
        
        // İl/İlçe için
        if ($var_name === 'il') {
            $region = get_post_meta($post->ID, '_kanews_auto_daily_region', true);
            return $region ? $region : '';
        }
        
        if ($var_name === 'ilce') {
            $city = get_post_meta($post->ID, '_kanews_auto_daily_city', true);
            return $city ? $city : '';
        }
        
        // Diğer veriler için region gerekli
        $region = get_post_meta($post->ID, '_kanews_auto_daily_region', true);
        $city = get_post_meta($post->ID, '_kanews_auto_daily_city', true);
        
        if (!$region) {
            return '';
        }
        
        // Sadece kullanılan değişken için veri çek
        switch ($var_name) {
            case 'hava':
                return $this->get_weather_data($region);
                
            case 'eczane':
                return $this->get_pharmacy_data($region, $city);
                
            case 'dolar':
                return $this->get_currency_data('USD');
                
            case 'euro':
                return $this->get_currency_data('EUR');
                
            case 'bitcoin':
                return $this->get_bitcoin_data();
                
            case 'ethereum':
                return $this->get_ethereum_data();
                
            case 'altin':
                return $this->get_gold_data();
                
            case 'ceyrek_altin':
                return $this->get_quarter_gold_data();
                
            case 'cumhuriyet_altin':
                return $this->get_republic_gold_data();
                
            case 'namaz':
                return $this->get_prayer_data($region, $city ? $city : $region);
                
            default:
                return '';
        }
    }
    
    /**
     * Frontend'de içerikteki shortcode'ları işle
     * Gün içinde direkt shortcode'ları işler, gün sonunda işlenmiş içerik kullanılır
     */
    public function process_shortcodes_in_content($content) {
        global $post;
        
        if (!$post || !get_post_meta($post->ID, '_kanews_auto_daily', true)) {
            return $content;
        }
        
        // İçerik zaten işlenmişse (gün sonunda kaydedilmiş) direkt döndür
        if (get_post_meta($post->ID, '_kanews_content_processed', true)) {
            return $content; // İçerik zaten işlenmiş halde
        }
        
        // Post'un tarihini al
        $post_date = get_post_meta($post->ID, '_kanews_auto_daily_date', true);
        if (!$post_date) {
            $post_date = date('Y-m-d', strtotime(get_the_date('Y-m-d', $post->ID)));
        }
        
        $today = date('Y-m-d');
        
        // Eğer post'un tarihi bugünden farklıysa (yani eski bir post), içeriği hemen işle ve kaydet
        // Böylece eski postlar için güncel veri çekilmeyecek
        if ($post_date !== $today) {
            // İçeriği işle (shortcode'ları gerçek değerlerle değiştir)
            $processed_content = do_shortcode($content);
            
            // Post içeriğini işlenmiş haline güncelle (shortcode'lar yerine gerçek veriler)
            wp_update_post(array(
                'ID' => $post->ID,
                'post_content' => $processed_content
            ));
            
            // Meta ekle ki tekrar kaydetmesin
            update_post_meta($post->ID, '_kanews_content_processed', true);
            
            // İşlenmiş içeriği döndür
            return $processed_content;
        }
        
        // Gün içinde shortcode'ları direkt işle
        return do_shortcode($content);
    }
    
    /**
     * Gün sonunda işlenmiş içeriği kaydetme kontrolü
     */
    public function maybe_save_processed_content() {
        // Sadece frontend'de çalışsın
        if (is_admin()) {
            return;
        }
        
        // Her saat başı kontrol et, ama sadece gün sonunda (23:55-23:59) kaydet
        $current_hour = (int) date('H');
        $current_minute = (int) date('i');
        
        // Gün sonunda kaydet (23:55-23:59 arası)
        if ($current_hour == 23 && $current_minute >= 55) {
            $today = date('Y-m-d');
            $last_save_date = get_option('kanews_processed_content_last_save_date');
            
            if ($last_save_date !== $today) {
                $this->save_processed_content();
                update_option('kanews_processed_content_last_save_date', $today);
            }
        }
    }
    
    /**
     * Gün sonunda işlenmiş içeriği kaydet
     */
    public function save_processed_content() {
        $today = date('Y-m-d');
        
        // Bugün oluşturulan ve işlenmemiş otomatik postları al
        $posts = get_posts(array(
            'post_type' => 'post',
            'posts_per_page' => -1,
            'meta_query' => array(
                'relation' => 'AND',
                array(
                    'key' => '_kanews_auto_daily',
                    'value' => true,
                    'compare' => '='
                ),
                array(
                    'key' => '_kanews_auto_daily_date',
                    'value' => $today,
                    'compare' => '='
                ),
                array(
                    'key' => '_kanews_content_processed',
                    'compare' => 'NOT EXISTS'
                )
            )
        ));
        
        foreach ($posts as $post) {
            // İçeriği işle (shortcode'ları gerçek değerlerle değiştir)
            $processed_content = do_shortcode($post->post_content);
            
            // Post içeriğini işlenmiş haline güncelle (shortcode'lar yerine gerçek veriler)
            wp_update_post(array(
                'ID' => $post->ID,
                'post_content' => $processed_content
            ));
            
            // Meta ekle ki tekrar kaydetmesin
            update_post_meta($post->ID, '_kanews_content_processed', true);
        }
    }
    
    /**
     * Değişkenleri değiştir
     */
    private function replace_variables($text, $date, $data) {
        // Tarih formatı
        $date_obj = new \DateTime($date);
        $formatted_date = $date_obj->format('d F Y');
        $formatted_date = str_replace(
            array('January', 'February', 'March', 'April', 'May', 'June', 
                  'July', 'August', 'September', 'October', 'November', 'December'),
            array('Ocak', 'Şubat', 'Mart', 'Nisan', 'Mayıs', 'Haziran',
                  'Temmuz', 'Ağustos', 'Eylül', 'Ekim', 'Kasım', 'Aralık'),
            $formatted_date
        );
        
        $replacements = array(
            '%%tarih%%' => $formatted_date,
            '%%il%%' => isset($data['region']) ? $data['region'] : '',
            '%%ilce%%' => isset($data['city']) ? $data['city'] : '',
            '%%hava%%' => isset($data['hava']) ? $data['hava'] : '',
            '%%eczane%%' => isset($data['eczane']) ? $data['eczane'] : '',
            '%%dolar%%' => isset($data['dolar']) ? $data['dolar'] : '',
            '%%euro%%' => isset($data['euro']) ? $data['euro'] : '',
            '%%bitcoin%%' => isset($data['bitcoin']) ? $data['bitcoin'] : '',
            '%%ethereum%%' => isset($data['ethereum']) ? $data['ethereum'] : '',
            '%%altin%%' => isset($data['altin']) ? $data['altin'] : '',
            '%%ceyrek_altin%%' => isset($data['ceyrek_altin']) ? $data['ceyrek_altin'] : '',
            '%%cumhuriyet_altin%%' => isset($data['cumhuriyet_altin']) ? $data['cumhuriyet_altin'] : '',
            '%%namaz%%' => isset($data['namaz']) ? $data['namaz'] : '',
        );
        
        return str_replace(array_keys($replacements), array_values($replacements), $text);
    }
    
    /**
     * AJAX: Tekil post silme
     */
    public function ajax_delete_daily_post() {
        check_ajax_referer('kanews_delete_post', 'nonce');
        
        if (!current_user_can('delete_posts')) {
            wp_send_json_error(array('message' => __('Yetkiniz yok.', 'kanews')));
        }
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $template_id = isset($_POST['template_id']) ? intval($_POST['template_id']) : 0;
        
        if (!$post_id || !$template_id) {
            wp_send_json_error(array('message' => __('Geçersiz parametreler.', 'kanews')));
        }
        
        // Template ID kontrolü
        $post_template_id = get_post_meta($post_id, '_kanews_auto_daily_template_id', true);
        if ($post_template_id != $template_id) {
            wp_send_json_error(array('message' => __('Bu yazı bu template\'e ait değil.', 'kanews')));
        }
        
        // Post'u sil
        $result = wp_delete_post($post_id, true);
        
        if ($result) {
            wp_send_json_success(array('message' => __('Yazı başarıyla silindi.', 'kanews')));
        } else {
            wp_send_json_error(array('message' => __('Yazı silinemedi.', 'kanews')));
        }
    }
    
    /**
     * AJAX: Toplu post silme
     */
    public function ajax_delete_daily_posts_bulk() {
        check_ajax_referer('kanews_delete_post', 'nonce');
        
        if (!current_user_can('delete_posts')) {
            wp_send_json_error(array('message' => __('Yetkiniz yok.', 'kanews')));
        }
        
        $post_ids = isset($_POST['post_ids']) ? array_map('intval', $_POST['post_ids']) : array();
        $template_id = isset($_POST['template_id']) ? intval($_POST['template_id']) : 0;
        
        if (empty($post_ids) || !$template_id) {
            wp_send_json_error(array('message' => __('Geçersiz parametreler.', 'kanews')));
        }
        
        $deleted = 0;
        $failed = 0;
        
        foreach ($post_ids as $post_id) {
            // Template ID kontrolü
            $post_template_id = get_post_meta($post_id, '_kanews_auto_daily_template_id', true);
            if ($post_template_id != $template_id) {
                $failed++;
                continue;
            }
            
            // Post'u sil
            $result = wp_delete_post($post_id, true);
            
            if ($result) {
                $deleted++;
            } else {
                $failed++;
            }
        }
        
        if ($deleted > 0) {
            $message = sprintf(__('%d yazı başarıyla silindi.', 'kanews'), $deleted);
            if ($failed > 0) {
                $message .= ' ' . sprintf(__('%d yazı silinemedi.', 'kanews'), $failed);
            }
            wp_send_json_success(array('message' => $message));
        } else {
            wp_send_json_error(array('message' => __('Hiçbir yazı silinemedi.', 'kanews')));
        }
    }
    
    /**
     * Cron job'ı planla
     */
    public static function schedule_cron() {
        // Günlük post oluşturma - Her gün 00:01'de
        if (!wp_next_scheduled('kanews_daily_content_cron')) {
            $timestamp = strtotime('tomorrow 00:01');
            wp_schedule_event($timestamp, 'daily', 'kanews_daily_content_cron');
        }
        
        // Gün sonunda işlenmiş içeriği kaydetme - Her gün 23:55'te
        if (!wp_next_scheduled('kanews_save_processed_content')) {
            $timestamp = strtotime('today 23:55');
            // Eğer bugün 23:55 geçtiyse, yarın 23:55'te planla
            if ($timestamp < time()) {
                $timestamp = strtotime('tomorrow 23:55');
            }
            wp_schedule_event($timestamp, 'daily', 'kanews_save_processed_content');
        }
    }
    
    /**
     * Cron job'ı temizle
     */
    public static function unschedule_cron() {
        wp_clear_scheduled_hook('kanews_daily_content_cron');
        wp_clear_scheduled_hook('kanews_save_processed_content');
    }
}

