<?php
namespace Kanews\Builder;
use Kanews\Helper_Wp;

class PageBuilder {

	private static $instance;

	/**
	 * Sidebar widget wrapper classes
	 */
	private const WIDGET_WRAPPER_FULL = '<div class="col"><section id="%1$s" class="kanews-section widget %2$s">';
	private const WIDGET_WRAPPER_CONTENT = '<div class="col"><section id="%1$s" class="kanews-section widget %2$s">';
	private const WIDGET_WRAPPER_SIDEBAR = '<div class="col-12"><section id="%1$s" class="kanews-section widget %2$s">';
	private const WIDGET_WRAPPER_BOTTOM = '<div class="col col-lg"><section id="%1$s" class="kanews-section widget %2$s">';
	
	private const TITLE_WRAPPER_H4 = '<div class="kanews-section-heading"><h4 class="kanews-section-headline"><span>';
	private const TITLE_WRAPPER_H5 = '<div class="kanews-section-heading"><h5 class="kanews-section-headline"><span>';
	private const TITLE_CLOSE_H4 = '</span></h4></div>';
	private const TITLE_CLOSE_H5 = '</span></h5></div>';

	/**
	 * Get singleton instance
	 * 
	 * @return self
	 */
	public static function getInstance() {
		if ( null === static::$instance ) {
			static::$instance = new static();
		}
		return static::$instance;
	}

	/**
	 * Constructor
	 */
	public function __construct() {
    
		// Templates
		add_action( 'kanews_header', [ $this, 'get_header' ] );
		add_action( 'kanews_footer', [ $this, 'get_footer' ] );

		// Home Builder
		add_action( 'kanews_page_builder_home', 'kanews_home_page_top', 15 );
		add_action( 'kanews_page_builder_home', 'kanews_home_page_content_start', 20 );
		add_action( 'kanews_page_builder_home', 'kanews_home_page_content', 25 );
		add_action( 'kanews_page_builder_home', 'kanews_home_page_sidebar', 30 );
		add_action( 'kanews_page_builder_home', 'kanews_home_page_content_end', 35 );
		add_action( 'kanews_page_builder_home', 'kanews_home_page_bottom', 40 );

		// Page Builder
		add_action( 'widgets_init', [ $this, 'page_builder' ], 20 );
		add_action( 'kanews_page_builder_custom', 'kanews_custom_page_top', 15 );
		add_action( 'kanews_page_builder_custom', 'kanews_custom_page_content_start', 20 );
		add_action( 'kanews_page_builder_custom', 'kanews_custom_page_content', 25 );
		add_action( 'kanews_page_builder_custom', 'kanews_custom_page_sidebar', 30 );
		add_action( 'kanews_page_builder_custom', 'kanews_custom_page_content_end', 35 );
		add_action( 'kanews_page_builder_custom', 'kanews_custom_page_bottom', 40 );

		// Cat Builder
		add_action( 'kanews_page_builder_cat', 'kanews_cat_page_top', 15 );
		add_action( 'kanews_page_builder_cat', 'kanews_cat_page_content_start', 20 );
		add_action( 'kanews_page_builder_cat', 'kanews_cat_page_content', 25 );
		add_action( 'kanews_page_builder_cat', 'kanews_cat_page_sidebar', 30 );
		add_action( 'kanews_page_builder_cat', 'kanews_cat_page_content_end', 35 );
		add_action( 'kanews_page_builder_cat', 'kanews_cat_page_bottom', 40 );
	}

	/**
	 * Get header template based on layout option
	 * 
	 * @return void
	 */
	public function get_header() {
		$layout = kanews_get_option( 'header-layout' ) ?: 'style-1';
		
		$header_map = [
			'style-4' => 'template-parts/header/header-4',
			'style-3' => 'template-parts/header/header-3',
			'style-2' => 'template-parts/header/header-2',
			'style-1' => 'template-parts/header/header-1',
		];

		$template = $header_map[ $layout ] ?? $header_map['style-1'];
		get_template_part( $template );
	}

	/**
	 * Get footer template based on layout option
	 * 
	 * @return void
	 */
	public function get_footer() {
		$layout = kanews_get_option( 'footer-layout' ) ?: 'style-1';
		
		$footer_map = [
			'style-1' => 'template-parts/footer/footer-1',
			'style-2' => 'template-parts/footer/footer-2',
		];

		$template = $footer_map[ $layout ] ?? $footer_map['style-1'];
		get_template_part( $template );
	}

	/**
	 * Register sidebars for page builder
	 * 
	 * @return void
	 */
	public function page_builder() {
		// Category Sidebars
		$categories = kanews_get_option( 'category-sidebar2' );
		if ( ! empty( $categories ) && is_array( $categories ) ) {
			foreach ( $categories as $cat_id ) {
				$cat = Helper_Wp::get_term_by( 'slug', $cat_id, 'category' );
				if ( ! is_wp_error( $cat ) && is_object( $cat ) && ! empty( $cat->name ) ) {
					$this->register_category_sidebars( $cat->name, $cat_id );
				}
			}
		}

		// Page Sidebars
		$custom_widgets = get_option( 'kanews_page_sidebars_widgets', [] );
		$get_custom_widgets = kanews_get_option( 'page-builder' );
		
		if ( ! empty( $get_custom_widgets ) && is_array( $get_custom_widgets ) ) {
			foreach ( $get_custom_widgets as $page_id ) {
				$name = get_the_title( $page_id );
				if ( ! empty( $name ) ) {
					$this->register_page_sidebars( $name, $page_id, 'all' );
				}
			}
		} elseif ( ! empty( $custom_widgets ) && is_array( $custom_widgets ) ) {
			foreach ( $custom_widgets as $post_id => $type ) {
				$name = get_the_title( $post_id );
				if ( ! empty( $name ) && ! empty( $type ) ) {
					$this->register_page_sidebars( $name, $post_id, $type );
				}
			}
		}

		// Köşe Yazıları
		$columnist_name = esc_html__( 'Köşe Yazıları', 'kanews-admin' );
		$this->register_single_sidebar( 
			sprintf( esc_html__( '%s: Sidebar', 'kanews-admin' ), $columnist_name ),
			'columnist-sidebar',
			self::WIDGET_WRAPPER_SIDEBAR,
			self::TITLE_WRAPPER_H5,
			self::TITLE_CLOSE_H5
		);

	}

	/**
	 * Register sidebars for a category
	 * 
	 * @param string $cat_name Category name
	 * @param string|int $cat_id Category ID or slug
	 * @return void
	 */
	private function register_category_sidebars( $cat_name, $cat_id ) {
		$name = esc_html( $cat_name );
		
		$this->register_single_sidebar(
			sprintf( esc_html__( '%s: Üst Tam Genişlik', 'kanews-admin' ), $name ),
			'home-top-full-' . $cat_id,
			self::WIDGET_WRAPPER_FULL,
			self::TITLE_WRAPPER_H4,
			self::TITLE_CLOSE_H4
		);

		$this->register_single_sidebar(
			sprintf( esc_html__( '%s: Sol Bölüm', 'kanews-admin' ), $name ),
			'home-content-' . $cat_id,
			self::WIDGET_WRAPPER_CONTENT,
			self::TITLE_WRAPPER_H4,
			self::TITLE_CLOSE_H4
		);

		$this->register_single_sidebar(
			sprintf( esc_html__( '%s: Sidebar', 'kanews-admin' ), $name ),
			'home-sidebar-' . $cat_id,
			self::WIDGET_WRAPPER_SIDEBAR,
			self::TITLE_WRAPPER_H4,
			self::TITLE_CLOSE_H4
		);

		$this->register_single_sidebar(
			sprintf( esc_html__( '%s: Alt Tam Genişlik', 'kanews-admin' ), $name ),
			'home-bottom-full-' . $cat_id,
			self::WIDGET_WRAPPER_FULL,
			self::TITLE_WRAPPER_H4,
			self::TITLE_CLOSE_H4
		);
	}

	/**
	 * Register sidebars for a page
	 * 
	 * @param string $name Page name
	 * @param int $page_id Page ID
	 * @param string $type Sidebar type ('all' or 'sidebar')
	 * @return void
	 */
	private function register_page_sidebars( $name, $page_id, $type ) {
		$escaped_name = esc_html( $name );
		
		if ( $type === 'all' ) {
			$this->register_single_sidebar(
				sprintf( esc_html__( '%s: Üst Tam Genişlik', 'kanews-admin' ), $escaped_name ),
				'home-top-full-' . $page_id,
				self::WIDGET_WRAPPER_FULL,
				self::TITLE_WRAPPER_H4,
				self::TITLE_CLOSE_H4
			);

			$this->register_single_sidebar(
				sprintf( esc_html__( '%s: Sol Bölüm', 'kanews-admin' ), $escaped_name ),
				'home-content-' . $page_id,
				self::WIDGET_WRAPPER_CONTENT,
				self::TITLE_WRAPPER_H4,
				self::TITLE_CLOSE_H4
			);

			$this->register_single_sidebar(
				sprintf( esc_html__( '%s: Sidebar', 'kanews-admin' ), $escaped_name ),
				'home-sidebar-' . $page_id,
				self::WIDGET_WRAPPER_SIDEBAR,
				self::TITLE_WRAPPER_H4,
				self::TITLE_CLOSE_H4
			);

			$this->register_single_sidebar(
				sprintf( esc_html__( '%s: Alt Tam Genişlik', 'kanews-admin' ), $escaped_name ),
				'home-bottom-full-' . $page_id,
				self::WIDGET_WRAPPER_BOTTOM,
				self::TITLE_WRAPPER_H4,
				self::TITLE_CLOSE_H4
			);
		} else {
			$this->register_single_sidebar(
				sprintf( esc_html__( '%s: Sidebar', 'kanews-admin' ), $escaped_name ),
				'home-sidebar-' . $page_id,
				self::WIDGET_WRAPPER_SIDEBAR,
				self::TITLE_WRAPPER_H5,
				self::TITLE_CLOSE_H5
			);
		}
	}

	/**
	 * Register Firma Rehberi sidebars
	 * 
	 * @param string $name Firma name
	 * @return void
	 */
	private function register_firma_sidebars( $name ) {
		$this->register_single_sidebar(
			sprintf( esc_html__( '%s: Üst Tam Genişlik', 'kanews-admin' ), $name ),
			'firma-top-full',
			self::WIDGET_WRAPPER_FULL,
			self::TITLE_WRAPPER_H4,
			self::TITLE_CLOSE_H4
		);

		$this->register_single_sidebar(
			sprintf( esc_html__( '%s: Sol Bölüm', 'kanews-admin' ), $name ),
			'firma-content',
			self::WIDGET_WRAPPER_CONTENT,
			self::TITLE_WRAPPER_H4,
			self::TITLE_CLOSE_H4
		);

		$this->register_single_sidebar(
			sprintf( esc_html__( '%s: Sidebar', 'kanews-admin' ), $name ),
			'firma-sidebar',
			self::WIDGET_WRAPPER_SIDEBAR,
			self::TITLE_WRAPPER_H4,
			self::TITLE_CLOSE_H4
		);
	}

	/**
	 * Register a single sidebar
	 * 
	 * @param string $name Sidebar name
	 * @param string $id Sidebar ID
	 * @param string $before_widget Widget wrapper HTML
	 * @param string $before_title Title wrapper HTML
	 * @param string $after_title Title close HTML
	 * @return void
	 */
	private function register_single_sidebar( $name, $id, $before_widget, $before_title, $after_title ) {
		register_sidebar( [
			'name' => $name,
			'id' => $id,
			'before_widget' => $before_widget,
			'after_widget' => '</section></div>',
			'before_title' => $before_title,
			'after_title' => $after_title,
		] );
	}

}