<?php
namespace Kanews;
use \stdClass;

class Category {

  private static $instance;

  public static function getInstance() {
    if ( null === static::$instance ) {
      static::$instance = new static();
    }
    return static::$instance;
  }

	public function __construct() {

		if( class_exists( 'WPSEO_Primary_Term' ) || class_exists('rank_math') )
		return;

		// Gutenberg için meta kaydı
		add_action( 'init', array( $this, 'kanews_register_meta' ) );
		
		// Klasik editör için metabox'u gizle
		add_action( 'add_meta_boxes', array( $this, 'kanews_remove_metabox' ), 99 );
		add_action( 'admin_head', array( $this, 'kanews_hide_metabox_css' ) );
		add_action( 'admin_footer', array( $this, 'kanews_add_primary_category_selector' ) );
		
		// Gutenberg için metabox ekle
		add_action( 'add_meta_boxes', array( $this, 'kanews_add_gutenberg_metabox' ) );
		
		add_action( 'save_post', array( $this, 'kanews_save_metabox' ) );
		// Gutenberg için REST API hook'u
		add_action( 'rest_after_insert_post', array( $this, 'kanews_save_metabox_rest' ), 10, 2 );
	}

	/**
	 * Metabox'u tamamen kaldır (sadece klasik editör için)
	 */
	public function kanews_remove_metabox() {
		$screen = get_current_screen();
		if ( $screen && 'post' === $screen->post_type ) {
			// Sadece klasik editörde kaldır (Gutenberg'te gösterilecek)
			if ( ! $screen->is_block_editor() ) {
				remove_meta_box( 'kanews_primary_category', 'post', 'side' );
				remove_meta_box( 'kanews_primary_category', 'post', 'normal' );
			}
		}
	}

	/**
	 * Metabox'u CSS ile gizle (sadece klasik editör için)
	 */
	public function kanews_hide_metabox_css() {
		$screen = get_current_screen();
		if ( $screen && 'post' === $screen->post_type ) {
			// Sadece klasik editörde gizle
			if ( ! $screen->is_block_editor() ) {
				echo '<style type="text/css">
					#kanews_primary_category,
					.postbox#kanews_primary_category,
					div#kanews_primary_category {
						display: none !important;
						visibility: hidden !important;
						height: 0 !important;
						overflow: hidden !important;
						margin: 0 !important;
						padding: 0 !important;
						border: none !important;
					}
				</style>';
			}
		}
	}

	/**
	 * Kategori meta box'ına birincil kategori seçici ekle
	 */
	public function kanews_add_primary_category_selector() {
		$screen = get_current_screen();
		global $post;
		
		if ( ! $screen || 'post' !== $screen->post_type ) {
			return;
		}

		$post_id = isset( $post->ID ) ? $post->ID : 0;
		$primary_category = '';
		
		if ( $post_id ) {
			$primary_category = get_post_meta( $post_id, 'kanews_primary_category', true );
		}

		// Nonce oluştur
		$nonce = wp_create_nonce( 'kanews_category_nonce' );
		?>
		<style type="text/css">
		.kanews-primary-category-btn {
			margin-left: 8px;
			padding: 2px 8px;
			font-size: 11px;
			line-height: 1.5;
			border: 1px solid #ccc;
			background: #f7f7f7;
			color: #555;
			cursor: pointer;
			border-radius: 3px;
			text-decoration: none;
			display: inline-block;
			vertical-align: middle;
		}
		.kanews-primary-category-btn:hover {
			background: #e5e5e5;
			border-color: #999;
		}
		.kanews-primary-category-btn.active {
			background: #0073aa;
			color: #fff;
			border-color: #0073aa;
			font-weight: 600;
		}
		.kanews-primary-category-btn.active:hover {
			background: #005a87;
			border-color: #005a87;
		}
		#categorydiv label,
		#taxonomy-category label {
			display: inline-flex;
			align-items: center;
		}
		</style>
		<script type="text/javascript">
		(function($) {
			$(document).ready(function() {
				var primaryCategory = '<?php echo esc_js( $primary_category ); ?>';
				var nonce = '<?php echo esc_js( $nonce ); ?>';

				// Hidden input'u form'a ekle veya güncelle
				if ($('#kanews_primary_category').length === 0) {
					$('#post').append('<input type="hidden" id="kanews_primary_category" name="kanews_primary_category" value="' + primaryCategory + '" />');
				} else {
					// Eğer zaten varsa, PHP'den gelen değerle güncelle (kayıt sonrası sayfa yenilendiğinde)
					$('#kanews_primary_category').val(primaryCategory);
				}
				if ($('#kanews_category_nonce_field').length === 0) {
					$('#post').append('<input type="hidden" id="kanews_category_nonce_field" name="kanews_category_nonce_field" value="' + nonce + '" />');
				}
				
				// PHP'den gelen değer zaten primaryCategory değişkeninde
				
				// Buton tıklaması sırasında yeniden oluşturmayı engellemek için flag
				var isButtonClicking = false;

				// Her kategori checkbox'ının yanına buton ekle (sadece seçili olanlar için)
				function addPrimaryButtons() {
					// Eğer buton tıklaması devam ediyorsa, yeniden oluşturma
					if (isButtonClicking) {
						return;
					}
					// Mevcut birincil kategoriyi önce hidden input'tan al, yoksa PHP'den gelen değeri kullan
					var currentPrimary = $('#kanews_primary_category').val();
					if (!currentPrimary || currentPrimary === '0' || currentPrimary === '') {
						currentPrimary = primaryCategory;
					}
					
					// Önce tüm mevcut butonları kaldır
					$('.kanews-primary-category-btn').remove();
					
					// Sadece seçili (checked) kategoriler için buton ekle
					$('#categorydiv input[type="checkbox"]:checked, #taxonomy-category input[type="checkbox"]:checked').each(function() {
						var $checkbox = $(this);
						var categoryId = String($checkbox.val()); // String'e çevir
						var $label = $checkbox.closest('label');
						
						// String karşılaştırması yap
						var isActive = String(categoryId) === String(currentPrimary);
						
						// Buton oluştur
						var btnClass = isActive ? 'kanews-primary-category-btn active' : 'kanews-primary-category-btn';
						var btnText = isActive ? '<?php echo esc_js( __( 'Birincil', 'kanews' ) ); ?>' : '<?php echo esc_js( __( 'Birincil Yap', 'kanews' ) ); ?>';
						
						var $btn = $('<a href="#" class="' + btnClass + '" data-category-id="' + categoryId + '">' + btnText + '</a>');
						
						// Buton tıklama olayı
						$btn.on('click', function(e) {
							e.preventDefault();
							e.stopPropagation(); // Event propagation'ı durdur
							
							// Flag'i set et
							isButtonClicking = true;
							
							var catId = $(this).data('category-id');
							
							// Tüm butonlardan active class'ını kaldır
							$('.kanews-primary-category-btn').removeClass('active').text('<?php echo esc_js( __( 'Birincil Yap', 'kanews' ) ); ?>');
							
							// Seçilen butona active class'ı ekle
							$(this).addClass('active').text('<?php echo esc_js( __( 'Birincil', 'kanews' ) ); ?>');
							
							// Hidden input'u güncelle
							$('#kanews_primary_category').val(catId);
							
							// primaryCategory değişkenini de güncelle
							primaryCategory = catId;
							
							// Flag'i kısa bir süre sonra sıfırla
							setTimeout(function() {
								isButtonClicking = false;
							}, 300);
							
							return false;
						});
						
						// Butonu label'a ekle
						$label.append($btn);
					});
					
					// Eğer birincil kategori seçili kategoriler arasında değilse, birincil kategoriyi sıfırla
					var primaryExists = false;
					$('#categorydiv input[type="checkbox"]:checked, #taxonomy-category input[type="checkbox"]:checked').each(function() {
						if (String($(this).val()) === String(currentPrimary)) {
							primaryExists = true;
							return false;
						}
					});
					
					if (!primaryExists && currentPrimary) {
						$('#kanews_primary_category').val('');
						primaryCategory = '';
					}
				}

				// İlk yüklemede butonları ekle (kategori listesinin yüklenmesini bekle)
				function initButtons() {
					// Kategori listesi yüklendi mi kontrol et
					if ($('#categorydiv input[type="checkbox"], #taxonomy-category input[type="checkbox"]').length > 0) {
						addPrimaryButtons();
					} else {
						// Henüz yüklenmemişse tekrar dene
						setTimeout(initButtons, 100);
					}
				}
				setTimeout(initButtons, 300);

				// MutationObserver ile kategori listesi değişikliklerini izle
				if (typeof MutationObserver !== 'undefined') {
					var observer = new MutationObserver(function(mutations) {
						// Eğer buton tıklaması devam ediyorsa, güncelleme yapma
						if (isButtonClicking) {
							return;
						}
						
						var shouldUpdate = false;
						mutations.forEach(function(mutation) {
							if (mutation.addedNodes.length > 0) {
								$(mutation.addedNodes).each(function() {
									// Buton eklenmesi değil, kategori checkbox'ı eklenmesi kontrolü
									if (!$(this).hasClass('kanews-primary-category-btn') && 
										!$(this).find('.kanews-primary-category-btn').length &&
										($(this).is('#categorydiv, #taxonomy-category') || 
										$(this).find('#categorydiv, #taxonomy-category').length > 0 ||
										$(this).closest('#categorydiv, #taxonomy-category').length > 0)) {
										shouldUpdate = true;
										return false;
									}
								});
							}
						});
						if (shouldUpdate) {
							setTimeout(addPrimaryButtons, 100);
						}
					});

					var categoryBox = $('#categorydiv, #taxonomy-category');
					if (categoryBox.length > 0) {
						observer.observe(categoryBox[0], {
							childList: true,
							subtree: true
						});
					}
				}

				// WordPress'in kategori checkbox'ları için özel event - seçim değiştiğinde butonları güncelle
				$(document).on('change', '#categorydiv input[type="checkbox"], #taxonomy-category input[type="checkbox"]', function(e) {
					// Eğer buton tıklamasından kaynaklanmıyorsa güncelle
					if (!$(e.target).hasClass('kanews-primary-category-btn') && !$(e.target).closest('.kanews-primary-category-btn').length) {
						setTimeout(addPrimaryButtons, 100);
					}
				});

				// WordPress AJAX sonrası için
				$(document).ajaxComplete(function() {
					// Eğer buton tıklaması devam ediyorsa, güncelleme yapma
					if (isButtonClicking) {
						return;
					}
					
					// Hidden input'tan güncel birincil kategoriyi al
					var savedPrimary = $('#kanews_primary_category').val();
					if (savedPrimary) {
						primaryCategory = savedPrimary;
					}
					setTimeout(addPrimaryButtons, 200);
				});

				// Sayfa yüklendiğinde ve kayıt sonrası için
				$(window).on('load', function() {
					// PHP'den gelen değeri tekrar al
					var phpPrimary = '<?php echo esc_js( $primary_category ); ?>';
					if (phpPrimary) {
						$('#kanews_primary_category').val(phpPrimary);
						primaryCategory = phpPrimary;
					}
					// Butonları güncelle (kategori listesi yüklendikten sonra)
					setTimeout(function() {
						addPrimaryButtons();
					}, 500);
				});
			});
		})(jQuery);
		</script>
		<?php
	}


	/**
	 * Gutenberg için meta kaydı
	 */
	public function kanews_register_meta() {
		register_meta( 'post', 'kanews_primary_category', array(
			'show_in_rest' => true,
			'single' => true,
			'type' => 'string',
			'auth_callback' => function() {
				return current_user_can( 'edit_posts' );
			}
		) );
	}

	/**
	 * Gutenberg için metabox ekle
	 */
	public function kanews_add_gutenberg_metabox() {
		$screen = get_current_screen();
		if ( $screen && 'post' === $screen->post_type && $screen->is_block_editor() ) {
			add_meta_box(
				'kanews_primary_category',
				__( 'Birincil Kategori', 'kanews' ),
				array( $this, 'kanews_render_gutenberg_metabox' ),
				'post',
				'side',
				'default'
			);
		}
	}

	/**
	 * Gutenberg metabox içeriğini render et
	 */
	public function kanews_render_gutenberg_metabox( $post ) {
		$primary_category = get_post_meta( $post->ID, 'kanews_primary_category', true );
		$categories = get_categories( array( 'hide_empty' => false ) );
		
		wp_nonce_field( 'kanews_category_nonce', 'kanews_category_nonce_field' );
		?>
		<div id="kanews-primary-category-metabox">
			<p>
				<label for="kanews_primary_category_select">
					<?php _e( 'Birincil kategori seçin:', 'kanews' ); ?>
				</label>
			</p>
			<select name="kanews_primary_category" id="kanews_primary_category_select" style="width: 100%;">
				<option value=""><?php _e( 'Seçiniz...', 'kanews' ); ?></option>
				<?php foreach ( $categories as $category ) : ?>
					<option value="<?php echo esc_attr( $category->term_id ); ?>" <?php selected( $primary_category, $category->term_id ); ?>>
						<?php echo esc_html( $category->name ); ?>
					</option>
				<?php endforeach; ?>
			</select>
			<p class="description">
				<?php _e( 'Bu kategori, yazının birincil kategorisi olarak kullanılacaktır.', 'kanews' ); ?>
			</p>
		</div>
		<?php
	}

	function kanews_save_metabox( $post_id ) {
		// Gutenberg'te meta REST API üzerinden otomatik kaydedilir
		// Bu fonksiyon sadece klasik editör için çalışır
		
		// Nonce kontrolü (sadece klasik editör için)
		if( isset( $_POST['kanews_category_nonce_field'] ) ) {
			if( ! wp_verify_nonce( $_POST['kanews_category_nonce_field'], 'kanews_category_nonce' ) ) {
				return;
			}
		}

		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		// Post tipi kontrolü
		if ( get_post_type( $post_id ) !== 'post' ) {
			return;
		}

		$primary_category = '';
		
		// POST'tan gelen değeri al (klasik editör)
		if ( isset( $_POST[ 'kanews_primary_category' ] ) ) {
			$primary_category = sanitize_text_field( $_POST[ 'kanews_primary_category' ] );
		}
		
		// Eğer birincil kategori seçilmemişse, yazının kategorilerinden ilkini al
		if ( empty( $primary_category ) || $primary_category === '0' ) {
			$categories = get_the_terms( $post_id, 'category' );
			if ( ! empty( $categories ) && ! is_wp_error( $categories ) ) {
				$primary_category = $categories[0]->term_id;
			}
		}
		
		// Meta veriyi kaydet
		if ( ! empty( $primary_category ) ) {
			update_post_meta( $post_id, 'kanews_primary_category', $primary_category );
		}
	}

	/**
	 * Gutenberg için REST API hook'u
	 */
	function kanews_save_metabox_rest( $post, $request ) {
		// Sadece post tipi için çalış
		if ( get_post_type( $post->ID ) !== 'post' ) {
			return;
		}

		if ( ! current_user_can( 'edit_post', $post->ID ) ) {
			return;
		}

		// Mevcut meta veriyi kontrol et
		$primary_category = get_post_meta( $post->ID, 'kanews_primary_category', true );
		
		// REST API'den gelen meta veriyi kontrol et
		$params = $request->get_params();
		if ( isset( $params['meta']['kanews_primary_category'] ) ) {
			$primary_category = sanitize_text_field( $params['meta']['kanews_primary_category'] );
		}
		
		// Eğer birincil kategori seçilmemişse, yazının kategorilerinden ilkini al
		if ( empty( $primary_category ) || $primary_category === '0' ) {
			$categories = get_the_terms( $post->ID, 'category' );
			if ( ! empty( $categories ) && ! is_wp_error( $categories ) ) {
				$primary_category = $categories[0]->term_id;
				// Meta veriyi kaydet
				update_post_meta( $post->ID, 'kanews_primary_category', $primary_category );
			}
		}
	}

  function kanews_build_lists() {
    $item = new stdClass();
    $args = array(
      'public' => true, // only get publically accessable post types
      '_builtin' => false // remove builtin post types
    );
    $item->post_type_list = get_post_types( $args, 'names' );
    $item->post_type_list['post'] = 'post';
    $item->categories_list = get_the_category();

    return $item;
  }
}
