<?php
defined( 'ABSPATH' ) || exit; // Exit if accessed directly

/**
 * Format font name for CSS
 * 
 * @param string $option Option name
 * @return string Formatted font name
 */
function kanews_global_font( $option ) {
	$font = kanews_get_option( $option );
	if ( empty( $font ) ) {
		return '';
	}

	$font_lower = strtolower( $font );
	$font_ucwords = ucwords( $font_lower, '_' );
	return str_replace( '_', ' ', $font_ucwords );
}

/**
 * Add color styling from theme
 * 
 * @return void
 */
function kanews_custom_css() {
	$styles = '';
	$single_styles = '';
	$terms = get_terms( [
		'taxonomy' => 'category',
		'hide_empty' => false,
	] );

	if ( is_wp_error( $terms ) ) {
		$terms = [];
	}

	$font_final = kanews_global_font( 'site-font' );

	wp_enqueue_style( 
		'kanews-theme', 
		KAN_TEMPLATE_URL . '/assets/css/theme.min.css', 
		[], 
		KAN_THEME_VER 
	);

	if ( is_singular() ) {
		wp_enqueue_style( 'kanews-theme-single' );
	}

	if ( kanews_get_option( 'mode', true ) 
		|| kanews_get_option( 'footer-skin', 'light' ) === 'dark' 
	) {
		wp_enqueue_style( 
			'kanews-dark-theme', 
			KAN_TEMPLATE_URL . '/assets/css/dark.min.css', 
			[], 
			KAN_THEME_VER 
		);
	}

	// Inline CSS
	$styles .= kanews_get_login_page_styles();
	
	// Lazy Load
	if ( kanews_get_option( 'lazy' ) && ! empty( kanews_get_option( 'lazy-logo' )['url'] ) ) {
		$styles .= sprintf(
			'.kanews-post-thumb:before,
			.kanews-slider-wrapper:not(.slick-initialized):before,
			.kanews-slide-thumb:before {
				background-image: url(%s);
			}',
			esc_url( kanews_get_option( 'lazy-logo' )['url'] )
		);
	}
	
	// Fonts
	if ( $font_final === 'System' ) {
		$styles .= 'body {font-family: -apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,Oxygen,Oxygen-Sans,Ubuntu,Cantarell,"Helvetica Neue","Open Sans",sans-serif !important; }';
	} elseif ( ! empty( $font_final ) ) {
		$styles .= sprintf( 'body {font-family: "%s", sans-serif !important;}' . "\n", esc_attr( $font_final ) );
	}

	// Single Fonts
	if ( is_singular() ) {
		$font_final2 = kanews_global_font( 'single-font' );
		if ( $font_final2 === 'System' ) {
			$single_styles .= '.entry-content-inner {font-family: -apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,Oxygen,Oxygen-Sans,Ubuntu,Cantarell,"Helvetica Neue","Open Sans",sans-serif !important; }';
		} elseif ( ! empty( $font_final2 ) ) {
			$single_styles .= sprintf( '.entry-content-inner {font-family: "%s", sans-serif !important;}' . "\n", esc_attr( $font_final2 ) );
		}
	}

	// Dark Logo
	$dark_logo = kanews_get_dark_logo();
	if ( ! empty( $dark_logo['url'] ) ) {
		$styles .= sprintf(
			'.dark-mode .site-header-logo img{ 
				content: url(%s);
				width: %spx;
			}' . "\n",
			esc_url( $dark_logo['url'] ),
			esc_attr( $dark_logo['width'] ?? '' )
		);
	}
	

	// All Settings
	$customizer = kanews_get_customizer_settings();

	$root = '';
	
	// Block Border
	if ( ! empty( $customizer['block-border-color'] ) ) {
		$color = esc_attr( $customizer['block-border-color'] );
		$root .= '--kan-border-color: ' . $color . ';';
		$root .= '--kan-border-color2: ' . $color . ';';
	}
	if ( ! empty( $customizer['block-border-radius'] ) ) {
		$root .= '--kan-block-radius: ' . esc_attr( $customizer['block-border-radius'] ) . 'px;';
		$styles .= '.kanews-section{overflow:hidden;}';
	}
	if ( $customizer['block-border'] === 'border' ) {
		$root .= '--kan-block-shadow: 0 0 0 1px var(--kan-border-color);';
	} elseif ( $customizer['block-border'] === 'none' ) {
		$root .= '--kan-block-shadow: none;';
	}
	
	// Header Border Radius
	if ( ! empty( $customizer['header-border-radius'] ) ) {
		$root .= '--kan-header-radius: ' . esc_attr( $customizer['header-border-radius'] ) . 'px;';
	}

	// Background
	$styles .= kanews_get_background_css( $customizer );
	
	// Container Width
	$styles .= kanews_get_container_width_css( $customizer );
	
	// Primary Color
	if ( ! empty( $customizer['primary'] ) ) {
		$root .= sprintf(
			'--wp--preset--color--primary: %s;
			--kan-theme-color: %s;
			',
			esc_attr( $customizer['primary'] ),
			esc_attr( $customizer['primary'] )
		);
	}

	// Sub Header
	$styles .= kanews_get_subheader_css( $customizer );

	// Header
	$styles .= kanews_get_header_css( $customizer );
	
	// Header Height
	$styles .= kanews_get_header_height_css( $customizer );

	if ( ! empty( $customizer['header-insights-btn-bg'] ) ) {
		$styles .= sprintf(
			'.header-kanews-insights-btn {background-color: %s}',
			esc_attr( $customizer['header-insights-btn-bg'] )
		);
	}

	// Footer
	$styles .= kanews_get_footer_css( $customizer );

	// Navbar
	$styles .= kanews_get_navbar_css( $customizer );

	// Breaking News Bar
	if ( ! empty( $customizer['bn-left-background'] ) ) {
		$bg = esc_attr( $customizer['bn-left-background'] );
		$styles .= sprintf( '.extra-sub .site-subheader {background-color: %s}', $bg );
		$styles .= '.extra-sub-action::after {background: linear-gradient(to right,transparent 0, ' . $bg . ' 100%)}';
	}
	if ( ! empty( $customizer['bn-left-color'] ) ) {
		$styles .= sprintf(
			'.extra-sub-action li a, .extra-sub-action .slick-arrow, .extra-sub .kanews-ticker-wrapper .kanews-ticker li>a, .extra-sub .kanews-ticker-wrapper .kanews-ticker-title, .extra-sub .kanews-ticker-wrapper .kanews-ticker-control>*{color: %s !important}',
			esc_attr( $customizer['bn-left-color'] )
		);
	}
	
	// Tab Bar
	if ( ! empty( $customizer['tab-bar-bg'] ) ) {
		$styles .= sprintf( '.kanews-tab-bar {background-color: %s !important}', esc_attr( $customizer['tab-bar-bg'] ) );
	}
	if ( ! empty( $customizer['tab-bar-color'] ) ) {
		$styles .= sprintf( '.kanews-tab-bar li>a, .kanews-tab-bar li, .kanews-tab-bar-icon {color: %s !important}', esc_attr( $customizer['tab-bar-color'] ) );
	}

	// Single Post Styles
	$single_styles .= kanews_get_single_post_styles( $customizer );
	
	// Post Title Typography
	$styles .= kanews_get_post_title_typography_css( $customizer );

	// Page Header
	$styles .= kanews_get_page_header_css( $customizer );

	// Big Menu
	$styles .= kanews_get_big_menu_css( $customizer );

	// Mobile Menu
	$styles .= kanews_get_mobile_menu_css( $customizer );

	// Back to top
	if ( ! empty( $customizer['backtotop-background'] ) ) {
		$styles .= sprintf( '.progress-wrap svg.progress-circle path {stroke: %s}', esc_attr( $customizer['backtotop-background'] ) );
	}
	if ( ! empty( $customizer['backtotop-color'] ) ) {
		$styles .= sprintf( '.progress-wrap::after {color: %s}', esc_attr( $customizer['backtotop-color'] ) );
	}

	// Main Category/Page Color
	$root .= kanews_get_primary_color_css();

	// Category Button Colors
	$styles .= kanews_get_category_colors_css( $terms );

	// KAI Audio Floating Player
	if ( \Kanews\Kai\Support\Options::get( 'kai-audio-enabled', false ) ) {
		$floating_primary = ! empty( $customizer['primary'] ) ? esc_attr( $customizer['primary'] ) : '#ef233c';
		$styles .= '
	.kanews-floating-audio{
		position:fixed;
		right:1rem;
		bottom:6rem;
		z-index:1040;
		display:flex;
		flex-direction:column;
		align-items:flex-end;
		gap:.5rem;
		opacity:1;
		transform:translateY(0);
		transition:opacity 0.3s ease,transform 0.3s ease;
	}
	.kanews-floating-audio__meta{
		background:#fff;
		color:#111;
		padding:.45rem .75rem;
		border-radius:.85rem;
		box-shadow:0 12px 30px rgba(15,23,42,.18);
		font-size:.85rem;
		max-width:240px;
		pointer-events:auto;
		position:relative;
	}
	.kanews-floating-audio__close{
		position:absolute;
		top:0;
		right:0;
		width:24px;
		height:24px;
		border-radius:50%;
		background:#fff;
		border:none;
		box-shadow:0 2px 8px rgba(0,0,0,.15);
		cursor:pointer;
		display:flex;
		align-items:center;
		justify-content:center;
		padding:0;
		z-index:10;
		pointer-events:auto;
		transition:transform .2s ease,background .2s ease;
		color:#666;
		margin-top:-8px;
		margin-right:-8px;
	}
	.kanews-floating-audio__close:hover{
		transform:scale(1.1);
		background:#f5f5f5;
		color:#333;
	}
	.kanews-floating-audio__close i{
		font-size:14px;
		line-height:1;
	}
	.dark-mode .kanews-floating-audio__close{
		background:rgba(15,23,42,.95);
		color:#f8fafc;
	}
	.dark-mode .kanews-floating-audio__close:hover{
		background:rgba(15,23,42,1);
	}
	.dark-mode .kanews-floating-audio__meta{
		background:var(--kan-surface-2);
		color:#f8fafc;
	}
	.kanews-floating-audio__slot{
		display:block;
		text-transform:uppercase;
		font-size:.72rem;
		letter-spacing:.08em;
		opacity:.7;
		margin-bottom:.1rem;
	}
	.kanews-floating-audio__title{
		display:block;
		font-weight:600;
		line-height:1.3;
	}
	.kanews-floating-audio__button{
		width:48px !important;
		height:48px !important;
		border-radius:50%;
		background:'.$floating_primary.';
		color:#fff;
		display:flex !important;
		align-items:center !important;
		justify-content:center !important;
		box-shadow:0 15px 35px rgba(0,0,0,.3);
		cursor:pointer !important;
		transition:transform .2s ease,box-shadow .2s ease;
		position:relative;
		pointer-events:auto;
	}
	.kanews-floating-audio__button:hover{
		transform:translateY(-2px);
		box-shadow:0 20px 40px rgba(0,0,0,.35);
	}
	.kanews-floating-audio[data-state="playing"] .kanews-floating-audio__button{
		box-shadow:0 0 0 8px rgba(255,255,255,.15),0 25px 50px rgba(0,0,0,.35);
	}
	.kanews-floating-audio .audio-icon svg{
		position:absolute;
		top:50%;
		left:50%;
		transform:translate(-50%,-50%);
		pointer-events:none;
	}
	.kanews-floating-audio .audio-icon svg .progress-circle-bg{
		fill:none;
		stroke:#fff;
		stroke-width:3;
		stroke-opacity:0.2;
		transform:rotate(-90deg);
		transform-origin:50% 50%;
	}
	.kanews-floating-audio .audio-icon svg .progress-circle-progress{
		fill:none;
		stroke:#fff;
		stroke-width:3;
		stroke-opacity:1;
		transform:rotate(-90deg);
		transform-origin:50% 50%;
		transition:stroke-dashoffset 0.1s linear;
	}
	.kanews-floating-audio__cta{
		font-size:.8rem;
		color:'.$floating_primary.';
		background:#fff;
		padding:.35rem .65rem;
		border-radius:999px;
		box-shadow:0 6px 20px rgba(15,23,42,.12);
		text-decoration:none;
		pointer-events:auto;
	}
	.tab-bar-active .kanews-floating-audio{
		bottom:3em
	}
	@media (max-width:768px){
		.kanews-floating-audio{
			right:.75rem;
			bottom:5rem;
		}
		.kanews-floating-audio__meta{
			max-width:200px;
		}
	}
	';
	}

	// Add root CSS variables
	if ( ! empty( $root ) ) {
		$styles .= ':root { ' . trim( $root ) . ' }';
	}

	wp_add_inline_style( 'kanews-theme', kan_minify_css( $styles ) );
	wp_add_inline_style( 'kanews-theme-single', kan_minify_css( $single_styles ) );
}

add_action( 'wp_enqueue_scripts', 'kanews_custom_css', 9999 );

/**
 * Get login page styles
 * 
 * @return string
 */
function kanews_get_login_page_styles() {
	if ( ! is_page() || ! kanews_get_option( 'login-left-background' ) ) {
		return '';
	}

	$bg = kanews_get_option( 'login-left-background' );
	$css = '.kanews-lr-page-left {';
	
	foreach ( $bg as $key => $value ) {
		if ( $key === 'background-image' && ! empty( $value['url'] ) ) {
			$css .= $key . ': url("' . esc_url( $value['url'] ) . '") !important;';
		} elseif ( $key !== 'background-image' && ! empty( $value ) ) {
			$css .= $key . ': ' . esc_attr( $value ) . ' !important;';
		}
	}
	
	$css .= '}';
	return $css;
}


/**
 * Get dark logo
 * 
 * @return array
 */
function kanews_get_dark_logo() {
	$dark_logo = kanews_get_option( 'logo-dark' );
	$dark_logo = ! empty( $dark_logo ) ? $dark_logo : [ 'url' => '' ];

	if ( is_category() || is_singular() ) {
		$term_id = is_category() 
			? (int) get_category( get_query_var( 'cat' ), false )->term_id 
			: (int) Kanews\Helper_Wp::primary_category_id( get_the_ID() );
		
		$logo_category = get_term_meta( $term_id, 'kanews_logo_dark', true );
		if ( ! empty( $logo_category['url'] ) ) {
			$dark_logo = $logo_category;
		}
	}
	
	return $dark_logo;
}

/**
 * Get customizer settings
 * 
 * @return array
 */
function kanews_get_customizer_settings() {
	return [
		'width' => kanews_get_option( 'site-container', '' ),
		'bg' => kanews_get_option( 'site-background', '' ),
		'primary' => kanews_get_option( 'primary-color' ),
		'header-bg' => kanews_get_option( 'header-color' ),
		'header-bg-gradient' => kanews_get_option( 'header-color-2' ),
		'header-insights-btn-bg' => kanews_get_option( 'subscribe-btn-color' ),
		'header-height' => kanews_get_option( 'height-height-tab' ),
		'header-border' => kanews_get_option( 'header-border-color' ),
		'header-border-radius' => kanews_get_option( 'header-border-radius' ),
		'header-btn-bg' => kanews_get_option( 'header-btn-bgcolor' ),
		'header-btn-color' => kanews_get_option( 'header-btn-color' ),
		'subheader-bg' => kanews_get_option( 'subheader-color' ),
		'subheader-color' => kanews_get_option( 'subheader-link-color' ),
		'footer-bg' => kanews_get_option( 'footer-color' ),
		'footer-bg-gradient' => kanews_get_option( 'footer-color-2' ),
		'navbar-bg' => kanews_get_option( 'navbar-color' ),
		'navbar-bg-gradient' => kanews_get_option( 'navbar-color-2' ),
		'navbarlink-color' => kanews_get_option( 'navbarlink-color' ),
		'navbarlinkhover-color' => kanews_get_option( 'navbarlinkhover-color' ),
		'navbar-typography' => kanews_get_option( 'navbar-typography' ),
		'mobilmenu-bg' => kanews_get_option( 'mobilmenu-color' ),
		'mobilmenu-bg-gradient' => kanews_get_option( 'mobilmenu-color-2' ),
		'mobilmenu-color' => kanews_get_option( 'mobilmenu-text-color' ),
		'mobilmenulink-color' => kanews_get_option( 'mobilmenulink-color' ),
		'pageheader-bg' => kanews_get_option( 'pageheader-color' ),
		'pageheader-bg-gradient' => kanews_get_option( 'pageheader-color-2' ),
		'pageheadertext-color' => kanews_get_option( 'pageheadertext-color' ),
		'bigmenu-bg' => kanews_get_option( 'bigmenu-color' ),
		'bigmenu-bg-gradient' => kanews_get_option( 'bigmenu-color-2' ),
		'bigmenulink-color' => kanews_get_option( 'bigmenulink-color' ),
		'backtotop-background' => kanews_get_option( 'backtotop-background' ),
		'backtotop-color' => kanews_get_option( 'backtotop-color' ),
		'single-postitle-color' => kanews_get_option( 'single-postitle-color' ),
		'single-postitle-typography' => kanews_get_option( 'single-postitle-typography' ),
		'single-postsubtitle-color' => kanews_get_option( 'single-postsubtitle-color' ),
		'single-postsubtitle-typography' => kanews_get_option( 'single-postsubtitle-typography' ),
		'single-postmeta-color' => kanews_get_option( 'single-postmeta-color' ),
		'single-postmeta-typography' => kanews_get_option( 'single-postmeta-typography' ),
		'single-postcontentlink-color' => kanews_get_option( 'single-postcontentlink-color' ),
		'single-postcontentlinkhover-color' => kanews_get_option( 'single-postcontentlinkhover-color' ),
		'single-postcontent-color' => kanews_get_option( 'single-postcontent-color' ),
		'single-postcontent-typography' => kanews_get_option( 'single-postcontent-typography' ),
		'postitle-typography' => kanews_get_option( 'postitle-typography' ),
		'bn-left-background' => kanews_get_option( 'bn-left-background' ),
		'bn-left-color' => kanews_get_option( 'bn-left-color' ),
		'bn-right-background' => kanews_get_option( 'bn-right-background' ),
		'bn-right-color' => kanews_get_option( 'bn-right-color' ),
		'tab-bar-bg' => kanews_get_option( 'tab-bar-bg' ),
		'tab-bar-color' => kanews_get_option( 'tab-bar-color' ),
		'block-border' => kanews_get_option( 'block-border', 'box_shadow' ),
		'block-border-color' => kanews_get_option( 'block-border-color', '' ),
		'block-border-radius' => kanews_get_option( 'block-border-radius', '' ),
	];
}


/**
 * Get background CSS
 * 
 * @param array $customizer Customizer settings
 * @return string
 */
function kanews_get_background_css( $customizer ) {
	$bg_array = $customizer['bg'] ?? [];
	
	if ( empty( $bg_array['background-color'] ) 
		&& empty( $bg_array['background-gradient-color'] ) 
		&& empty( $bg_array['background-image']['url'] ) 
	) {
		return '';
	}

	$css = '.site-wrapper {';
	$has_gradient = ! empty( $bg_array['background-color'] ) && ! empty( $bg_array['background-gradient-color'] );
	$gradient_direction = $bg_array['background-gradient-direction'] ?? 'to right';

	if ( $has_gradient ) {
		$css .= sprintf(
			'background: linear-gradient(%s, %s, %s);',
			esc_attr( $gradient_direction ),
			esc_attr( $bg_array['background-color'] ),
			esc_attr( $bg_array['background-gradient-color'] )
		);
	} elseif ( ! empty( $bg_array['background-color'] ) ) {
		$css .= 'background: ' . esc_attr( $bg_array['background-color'] ) . ';';
	}

	if ( ! empty( $bg_array['background-image']['url'] ) ) {
		$css .= 'background-image: url(' . esc_url( $bg_array['background-image']['url'] ) . ');';
	}

	foreach ( $bg_array as $key => $value ) {
		if ( in_array( $key, [ 'background-image', 'background-color', 'background-gradient-color', 'background-gradient-direction' ], true ) ) {
			continue;
		}
		if ( ! empty( $value ) ) {
			$css .= $key . ': ' . esc_attr( $value ) . ';';
		}
	}

	$css .= '}';
	return $css;
}

/**
 * Get container width CSS
 * 
 * @param array $customizer Customizer settings
 * @return string
 */
function kanews_get_container_width_css( $customizer ) {
	if ( empty( $customizer['width']['width'] ) 
		|| ! is_numeric( $customizer['width']['width'] ) 
		|| (int) $customizer['width']['width'] === 1200
	) {
		return '';
	}

	return sprintf(
		'@media (min-width: 992px) {
			.container {
				max-width: %s%s
			}
		}',
		esc_attr( $customizer['width']['width'] ),
		esc_attr( $customizer['width']['unit'] ?? 'px' )
	);
}

/**
 * Get subheader CSS
 * 
 * @param array $customizer Customizer settings
 * @return string
 */
function kanews_get_subheader_css( $customizer ) {
	$css = '';
	
	if ( ! empty( $customizer['subheader-bg'] ) ) {
		$css .= sprintf(
			'html:not(.dark-mode) .site-header-wrapper .site-subheader {background-color: %s}',
			esc_attr( $customizer['subheader-bg'] )
		);
	}
	
	if ( ! empty( $customizer['subheader-color'] ) ) {
		$css .= sprintf(
			'html:not(.dark-mode) .site-header-wrapper .site-subheader, html:not(.dark-mode) .site-header-wrapper .site-subheader * {color: %s !important}',
			esc_attr( $customizer['subheader-color'] )
		);
	}
	
	return $css;
}

/**
 * Get header CSS
 * 
 * @param array $customizer Customizer settings
 * @return string
 */
function kanews_get_header_css( $customizer ) {
	$css = '';
	
	if ( ! empty( $customizer['header-bg-gradient'] ) && ! empty( $customizer['header-bg'] ) ) {
		$css .= sprintf(
			'.site-header .site-header-wrapper {background: linear-gradient(to right, %s, %s)}',
			esc_attr( $customizer['header-bg'] ),
			esc_attr( $customizer['header-bg-gradient'] )
		);
	} elseif ( ! empty( $customizer['header-bg'] ) ) {
		$css .= sprintf(
			'.site-header .site-header-wrapper {background-color: %s}',
			esc_attr( $customizer['header-bg'] )
		);
	}
	
	if ( ! empty( $customizer['header-btn-bg'] ) ) {
		$css .= sprintf(
			'html:not(.dark-mode) .site-header .header-btn-icon, html:not(.dark-mode) .header-btn, html:not(.dark-mode) .site-header button {background-color: %s}',
			esc_attr( $customizer['header-btn-bg'] )
		);
	}
	
	if ( ! empty( $customizer['header-btn-color'] ) ) {
		$css .= sprintf(
			'html:not(.dark-mode) .site-header .header-btn-icon, html:not(.dark-mode) .header-btn, html:not(.dark-mode) .site-header button {color: %s}',
			esc_attr( $customizer['header-btn-color'] )
		);
	}
	
	if ( ! empty( $customizer['header-border'] ) ) {
		$css .= sprintf(
			'html:not(.dark-mode) .site-header {--kan-border-color: %s}',
			esc_attr( $customizer['header-border'] )
		);
	}
	
	return $css;
}

/**
 * Get header height CSS
 * 
 * @param array $customizer Customizer settings
 * @return string
 */
function kanews_get_header_height_css( $customizer ) {
	$css = '';
	
	// Desktop height
	if ( ! empty( $customizer['header-height']['desktop']['height'] ) 
		&& is_numeric( $customizer['header-height']['desktop']['height'] ) 
	) {
		$extra_height = 0;
		if ( kanews_get_option( 'subheader-switch' ) && kanews_get_option( 'subnavbar-left-bn' ) ) {
			$extra_height = 80;
		} elseif ( kanews_get_option( 'subheader-switch' ) ) {
			$extra_height = 40;
		} elseif ( kanews_get_option( 'subnavbar-left-bn' ) ) {
			$extra_height = 40;
		}

		$height = $customizer['header-height']['desktop']['height'];
		$unit = $customizer['header-height']['desktop']['unit'] ?? 'px';
		$calculated_height = $height + $extra_height;
		
		$css .= sprintf(
			'@media (min-width: 992px) {
				.site-header .site-header-top, .site-header .site-header-top .site-row {
					height: %s%s !important;
				}
				.site-head-2 {
					height: %s%s !important;
				}
			}',
			esc_attr( $height ),
			esc_attr( $unit ),
			esc_attr( $calculated_height ),
			esc_attr( $unit )
		);
	}
	
	// Mobile height
	if ( ! empty( $customizer['header-height']['mobile']['height'] ) 
		&& is_numeric( $customizer['header-height']['mobile']['height'] )
		&& (int) $customizer['header-height']['mobile']['height'] !== 64
	) {
		$height = $customizer['header-height']['mobile']['height'];
		$unit = $customizer['header-height']['mobile']['unit'] ?? 'px';
		
		$css .= sprintf(
			'@media (max-width: 992px) {
				.site-header, .site-header .site-header-top, .site-header .site-header-top .site-row {
					height: %s%s !important;
				}
				.site-head-2 {
					height: %s%s !important;
				}
				.site-head-2 .site-navbar, .site-head-2 .site-navbar .navbar-row, .site-head-2 .site-navbar .site-navbar-nav>li {
					height: %s%s !important;
				}
				.site-head-2 .site-navbar .site-navbar-nav>li>a {
					line-height: %s%s !important;
				}
			}',
			esc_attr( $height ),
			esc_attr( $unit ),
			esc_attr( $height ),
			esc_attr( $unit ),
			esc_attr( $height ),
			esc_attr( $unit ),
			esc_attr( $height ),
			esc_attr( $unit )
		);
	}
	
	return $css;
}

/**
 * Get footer CSS
 * 
 * @param array $customizer Customizer settings
 * @return string
 */
function kanews_get_footer_css( $customizer ) {
	$css = '';
	
	if ( ! empty( $customizer['footer-bg-gradient'] ) && ! empty( $customizer['footer-bg'] ) ) {
		$css .= sprintf(
			'#footer.dark-mode,.site-footer.light-mode {background: linear-gradient(to right, %s, %s)}',
			esc_attr( $customizer['footer-bg'] ),
			esc_attr( $customizer['footer-bg-gradient'] )
		);
	} elseif ( ! empty( $customizer['footer-bg'] ) ) {
		$css .= sprintf(
			'#footer {--kan-surface-2: %s}',
			esc_attr( $customizer['footer-bg'] )
		);
		$css .= sprintf(
			'#footer {--kan-bg-white-color: %s}',
			esc_attr( $customizer['footer-bg'] )
		);
	}
	
	return $css;
}

/**
 * Get navbar CSS
 * 
 * @param array $customizer Customizer settings
 * @return string
 */
function kanews_get_navbar_css( $customizer ) {
	$css = '';
	
	if ( ! empty( $customizer['navbar-bg-gradient'] ) && ! empty( $customizer['navbar-bg'] ) ) {
		$css .= sprintf(
			'.site-navbar-wrapper, .site-navbar .site-navbar-nav li .sub-menu, .sie-head-3 .site-header .header-skin-light {background: linear-gradient(to right, %s, %s) !important}',
			esc_attr( $customizer['navbar-bg'] ),
			esc_attr( $customizer['navbar-bg-gradient'] )
		);
	} elseif ( ! empty( $customizer['navbar-bg'] ) ) {
		$css .= sprintf(
			'.site-navbar-wrapper, .site-navbar .site-navbar-nav li .sub-menu, .site-head-3 .site-header .header-skin-light {background-color: %s !important}',
			esc_attr( $customizer['navbar-bg'] )
		);
	}
	
	if ( ! empty( $customizer['navbarlink-color'] ) ) {
		$color = esc_attr( $customizer['navbarlink-color'] );
		$css .= sprintf( '.site-navbar .site-navbar-nav li a {color: %s}', $color );
		$css .= sprintf( '.site-head-3 button {color: %s !important}', $color );
	}
	
	if ( ! empty( $customizer['navbarlinkhover-color'] ) ) {
		$color = esc_attr( $customizer['navbarlinkhover-color'] );
		$css .= sprintf(
			'.site-navbar .site-navbar-nav li a:hover,.site-navbar .site-navbar-nav li .sub-menu li.current-category-ancestor>a, .site-navbar .site-navbar-nav li .sub-menu li.current-category-parent>a, .site-navbar .site-navbar-nav li .sub-menu li.current-menu-ancestor>a, .site-navbar .site-navbar-nav li .sub-menu li.current-menu-item>a, .site-navbar .site-navbar-nav li .sub-menu li.current-menu-parent>a, .site-navbar .site-navbar-nav>li.current-category-ancestor>a, .site-navbar .site-navbar-nav>li.current-category-parent>a, .site-navbar .site-navbar-nav>li.current-menu-ancestor>a, .site-navbar .site-navbar-nav>li.current-menu-item>a, .site-navbar .site-navbar-nav>li.current-menu-parent>a {color: %s}',
			$color
		);
		$css .= sprintf( '.site-navbar .site-navbar-nav>li:after {background-color: %s}', $color );
	}
	
	// Navbar Typography
	if ( ! empty( $customizer['navbar-typography'] ) ) {
		$typography = $customizer['navbar-typography'];
		$has_typography = ! empty( $typography['text-transform'] ) 
			|| ! empty( $typography['font-size'] ) 
			|| ! empty( $typography['font-weight'] ) 
			|| ! empty( $typography['type'] );
		
		if ( $has_typography ) {
			$css .= '.site-navbar .site-navbar-nav li a {';
			foreach ( $typography as $key => $value ) {
				if ( empty( $value ) || $key === 'unit' ) {
					continue;
				}
				
				if ( in_array( $key, [ 'text-transform', 'font-weight' ], true ) ) {
					$css .= $key . ': ' . esc_attr( $value ) . ' !important;';
				} else {
					$css .= $key . ': ' . esc_attr( $value ) . 'px !important;';
				}
			}
			$css .= '}';
		}
	}
	
	return $css;
}


/**
 * Get single post styles
 * 
 * @param array $customizer Customizer settings
 * @return string
 */
function kanews_get_single_post_styles( $customizer ) {
	$css = '';
	
	// Title
	if ( ! empty( $customizer['single-postitle-color'] ) ) {
		$css .= sprintf(
			'.kanews-article-title {color: %s}',
			esc_attr( $customizer['single-postitle-color'] )
		);
	}
	
	$css .= kanews_get_typography_css( 
		$customizer['single-postitle-typography'] ?? [], 
		'.kanews-article-title' 
	);
	
	// Excerpt
	if ( ! empty( $customizer['single-postsubtitle-color'] ) ) {
		$css .= sprintf(
			'.kanews-article-title+p {color: %s}',
			esc_attr( $customizer['single-postsubtitle-color'] )
		);
	}
	
	$css .= kanews_get_typography_css( 
		$customizer['single-postsubtitle-typography'] ?? [], 
		'.kanews-article-title+p' 
	);
	
	// Meta
	if ( ! empty( $customizer['single-postmeta-color'] ) ) {
		$css .= sprintf(
			'.kanews-article-meta {color: %s}',
			esc_attr( $customizer['single-postmeta-color'] )
		);
	}
	
	$css .= kanews_get_typography_css( 
		$customizer['single-postmeta-typography'] ?? [], 
		'.kanews-article-meta' 
	);
	
	// Content
	if ( ! empty( $customizer['single-postcontent-color'] ) ) {
		$css .= sprintf(
			'.kanews-article-content li, .kanews-article-content p, .kanews-article-content {color: %s}',
			esc_attr( $customizer['single-postcontent-color'] )
		);
	}
	
	if ( ! empty( $customizer['single-postcontentlink-color'] ) ) {
		$css .= sprintf(
			'.entry-content-inner a {color: %s}',
			esc_attr( $customizer['single-postcontentlink-color'] )
		);
	}
	
	if ( ! empty( $customizer['single-postcontentlinkhover-color'] ) ) {
		$css .= sprintf(
			'.entry-content-inner a:hover {color: %s}',
			esc_attr( $customizer['single-postcontentlinkhover-color'] )
		);
	}
	
	$css .= kanews_get_typography_css( 
		$customizer['single-postcontent-typography'] ?? [], 
		'.kanews-article-content li, .kanews-article-content p, .kanews-article-content',
		false
	);
	
	return $css;
}

/**
 * Get typography CSS
 * 
 * @param array $typography Typography settings
 * @param string $selector CSS selector
 * @param bool $important Use !important
 * @return string
 */
function kanews_get_typography_css( $typography, $selector, $important = true ) {
	if ( empty( $typography ) ) {
		return '';
	}

	$css = '@media (min-width: 992px){' . $selector . ' {';
	
	foreach ( $typography as $key => $value ) {
		if ( empty( $value ) || $key === 'unit' ) {
			continue;
		}
		
		if ( in_array( $key, [ 'text-transform', 'text-decoration', 'font-weight' ], true ) ) {
			$css .= $key . ': ' . esc_attr( $value ) . ';';
		} else {
			$important_str = $important ? ' !important' : '';
			$css .= $key . ': ' . esc_attr( $value ) . 'px' . $important_str . ';';
		}
	}
	
	$css .= '}}';
	return $css;
}

/**
 * Get post title typography CSS
 * 
 * @param array $customizer Customizer settings
 * @return string
 */
function kanews_get_post_title_typography_css( $customizer ) {
	return kanews_get_typography_css( 
		$customizer['postitle-typography'] ?? [], 
		'.kanews-post-headline' 
	);
}

/**
 * Get page header CSS
 * 
 * @param array $customizer Customizer settings
 * @return string
 */
function kanews_get_page_header_css( $customizer ) {
	$css = '';
	
	if ( ! empty( $customizer['pageheader-bg-gradient'] ) && ! empty( $customizer['pageheader-bg'] ) ) {
		$css .= sprintf(
			'.kanews-page-header {background: linear-gradient(to right, %s, %s)}',
			esc_attr( $customizer['pageheader-bg'] ),
			esc_attr( $customizer['pageheader-bg-gradient'] )
		);
	} elseif ( ! empty( $customizer['pageheader-bg'] ) ) {
		$css .= sprintf(
			'.kanews-page-header {background-color: %s}',
			esc_attr( $customizer['pageheader-bg'] )
		);
	}
	
	if ( ! empty( $customizer['pageheadertext-color'] ) ) {
		$css .= sprintf(
			'.kanews-page-header, .kanews-page-header .kanews-page-title {color: %s}',
			esc_attr( $customizer['pageheadertext-color'] )
		);
	}
	
	return $css;
}

/**
 * Get big menu CSS
 * 
 * @param array $customizer Customizer settings
 * @return string
 */
function kanews_get_big_menu_css( $customizer ) {
	$css = '';
	
	if ( ! empty( $customizer['bigmenu-bg-gradient'] ) && ! empty( $customizer['bigmenu-bg'] ) ) {
		$css .= sprintf(
			'#site-menu-block {background: linear-gradient(to right, %s, %s)}',
			esc_attr( $customizer['bigmenu-bg'] ),
			esc_attr( $customizer['bigmenu-bg-gradient'] )
		);
	} elseif ( ! empty( $customizer['bigmenu-bg'] ) ) {
		$css .= sprintf(
			'#site-menu-block {background-color: %s}',
			esc_attr( $customizer['bigmenu-bg'] )
		);
	}
	
	if ( ! empty( $customizer['bigmenulink-color'] ) ) {
		$css .= sprintf(
			'#site-menu-block .site-menu-block>li a {color: %s}',
			esc_attr( $customizer['bigmenulink-color'] )
		);
	}
	
	return $css;
}

/**
 * Get mobile menu CSS
 * 
 * @param array $customizer Customizer settings
 * @return string
 */
function kanews_get_mobile_menu_css( $customizer ) {
	$css = '';
	
	if ( ! empty( $customizer['mobilmenu-bg-gradient'] ) && ! empty( $customizer['mobilmenu-bg'] ) ) {
		$css .= sprintf(
			'#site-mobile-menu .site-mobile-menu-wrapper{background: linear-gradient(to right, %s, %s)}',
			esc_attr( $customizer['mobilmenu-bg'] ),
			esc_attr( $customizer['mobilmenu-bg-gradient'] )
		);
	} elseif ( ! empty( $customizer['mobilmenu-bg'] ) ) {
		$css .= sprintf(
			'#site-mobile-menu .site-mobile-menu-wrapper {background-color: %s}',
			esc_attr( $customizer['mobilmenu-bg'] )
		);
	}
	
	if ( ! empty( $customizer['mobilmenulink-color'] ) ) {
		$css .= sprintf(
			'#site-mobile-menu .site-mobile-menu-nav>ul>li>a, #site-mobile-menu .site-mobile-menu-footer {color: %s}',
			esc_attr( $customizer['mobilmenulink-color'] )
		);
	}
	
	if ( ! empty( $customizer['mobilmenu-color'] ) ) {
		$css .= sprintf(
			'#site-mobile-menu,#site-mobile-menu .site-mobile-menu-footer p {color: %s}',
			esc_attr( $customizer['mobilmenu-color'] )
		);
	}
	
	return $css;
}


/**
 * Get primary color CSS
 * 
 * @return string
 */
function kanews_get_primary_color_css() {
	$root = '';
	
	if ( is_category() ) {
		$cat = get_query_var( 'cat' );
		$cat_color = get_term_meta( $cat, '_category_color', true );
		if ( ! empty( $cat_color ) ) {
			$root .= sprintf(
				'--wp--preset--color--primary: #%s;
				--kan-theme-color: #%s;
				',
				esc_attr( $cat_color ),
				esc_attr( $cat_color )
			);
		}
	} elseif ( is_single() && kanews_get_option( 'switch-primary-color', true ) ) {
		$cat = (int) Kanews\Helper_Wp::primary_category_id( get_the_ID() );
		$cat_color = get_term_meta( $cat, '_category_color', true );
		if ( ! empty( $cat_color ) && get_post_meta( get_the_ID(), 'primary_cat_color', true ) !== 'off' ) {
			$root .= sprintf(
				'--wp--preset--color--primary: #%s;
				--kan-theme-color: #%s;
				',
				esc_attr( $cat_color ),
				esc_attr( $cat_color )
			);
		}
	} elseif ( is_page() ) {
		$page = get_queried_object_id();
		$page_color = get_post_meta( $page, '_page_color', true );
		if ( ! empty( $page_color ) ) {
			$root .= sprintf(
				'--wp--preset--color--primary: %s;
				--kan-theme-color: %s;
				',
				esc_attr( $page_color ),
				esc_attr( $page_color )
			);
		}
	}
	
	return $root;
}

/**
 * Get category colors CSS
 * 
 * @param array $terms Terms array
 * @return string
 */
function kanews_get_category_colors_css( $terms ) {
	if ( empty( $terms ) || ! is_array( $terms ) ) {
		return '';
	}

	$css = '';
	
	foreach ( $terms as $term ) {
		$cat_color = get_term_meta( $term->term_id, '_category_color', true );
		if ( ! empty( $cat_color ) ) {
			$css .= sprintf(
				'.kanews-category-%d.kanews-category {--kan-theme-color: #%s !important}',
				(int) $term->term_id,
				esc_attr( $cat_color )
			);
		}
	}
	
	return $css;
}	

/**
 * Add theme color meta tags
 * 
 * @return void
 */
add_action( 'wp_head', 'kanews_wp_head', 999 );
if ( ! function_exists( 'kanews_wp_head' ) ) {
	function kanews_wp_head() {
		$primary_color = kanews_get_option( 'primary-color' );

		if ( kanews_get_option( 'switch-primary-color', true ) ) {
			if ( is_category() ) {
				$cat = get_query_var( 'cat' );
				$cat_color = get_term_meta( $cat, '_category_color', true );
				if ( ! empty( $cat_color ) ) {
					$primary_color = '#' . esc_attr( $cat_color );
				}
			} elseif ( is_single() ) {
				$cat = (int) Kanews\Helper_Wp::primary_category_id( get_the_ID() );
				$cat_color = get_term_meta( $cat, '_category_color', true );
				if ( ! empty( $cat_color ) 
					&& get_post_meta( get_the_ID(), 'primary_cat_color', true ) !== 'off' 
				) {
					$primary_color = '#' . esc_attr( $cat_color );
				}
			} elseif ( is_page() ) {
				$page = get_queried_object_id();
				$page_color = get_post_meta( $page, '_page_color', true );
				if ( ! empty( $page_color ) ) {
					$primary_color = esc_attr( $page_color );
				}
			}
		}
		
		$primary_color = ! empty( $primary_color ) ? esc_attr( $primary_color ) : '#e50914';
		
		printf(
			'<meta name="theme-color" content="%s" />
			<meta name="msapplication-navbutton-color" content="%s" />
			<meta name="apple-mobile-web-app-status-bar-style" content="%s" />',
			$primary_color,
			$primary_color,
			$primary_color
		);
	}
}


/**
 * Include Site Font
 * 
 * @return void
 */
add_action( 'wp_footer', 'dynamic_scripts' );
if ( ! function_exists( 'dynamic_scripts' ) ) {
	function dynamic_scripts() {
		$font_final = kanews_global_font( 'site-font' );
		$skip_fonts = false;
		$font_single = '';
		
		if ( empty( $font_final ) ) {
			$font_final = 'Inter';
		} elseif ( $font_final === 'System' ) {
			$skip_fonts = true;
			if ( ! is_singular() ) {
				return;
			}
		}
		
		$font_main = esc_js( $font_final ) . ':regular,500,600,700:latin&display=swap';
		
		if ( is_singular() || is_admin() ) {
			$font_final2 = kanews_global_font( 'single-font' );
			if ( empty( $font_final2 ) ) {
				$font_final2 = 'Inter';
			} elseif ( $font_final2 === 'System' ) {
				if ( $skip_fonts ) {
					return;
				}
			}
			
			if ( $font_final2 !== $font_final && $font_final2 !== 'System' ) {
				$font_single = esc_js( $font_final2 ) . ':regular,500,600,700:latin&display=swap';
			}
		}
		
		?>
		<script>
		(function() {
			var connection = navigator.connection || navigator.mozConnection || navigator.webkitConnection;
			if ( typeof connection !== 'undefined' && (/\slow-2g|2g/.test(connection.effectiveType))) {
				console.warn( 'Slow Connection Google Fonts Disabled' );
				return;
			}
			
			var fontFamilies = ['<?php echo $font_main; ?>'];
			<?php if ( ! empty( $font_single ) ) : ?>
			fontFamilies.push('<?php echo $font_single; ?>');
			<?php endif; ?>
			
			WebFontConfig = {
				google: {
					families: fontFamilies
				}
			};

			var wf = document.createElement('script');
			wf.src = '//ajax.googleapis.com/ajax/libs/webfont/1.6.26/webfont.js';
			wf.type = 'text/javascript';
			wf.defer = true;
			var s = document.getElementsByTagName('script')[0];
			s.parentNode.insertBefore(wf, s);
		})();
		</script>
		<?php
	}
}