<?php
defined( 'ABSPATH' ) || exit; // Exit if accessed directly
/**
 * Template Name: Kullanıcı Paneli
 * @url kanthemes.com
 */
__( 'Kullanıcı Paneli', 'kanews' );

/**
 * Helper function to determine post status based on user role
 * 
 * @param int $user_id The user ID
 * @return string The post status
 */
function kanews_get_post_status_by_role($user_id) {
    $post_status_subscriber = kanews_get_option('post-submit-status') != '' ? kanews_get_option('post-submit-status') : 'pending';
    $post_status_columist = kanews_get_option('post-submit-status-columist') != '' ? kanews_get_option('post-submit-status-columist') : 'pending';
    $post_status_contributor = kanews_get_option('post-submit-status-contributor') != '' ? kanews_get_option('post-submit-status-contributor') : 'pending';
    $post_status_author = kanews_get_option('post-submit-status-author') != '' ? kanews_get_option('post-submit-status-author') : 'pending';
    $post_status_editor = kanews_get_option('post-submit-status-editor') != '' ? kanews_get_option('post-submit-status-editor') : 'pending';

    $user_meta = get_userdata($user_id);
    $role = isset($user_meta->roles[0]) ? $user_meta->roles[0] : '';

    switch($role) {
        case 'administrator':
            return 'publish';
        case 'editor':
            return $post_status_editor;
        case 'author':
            return $post_status_author;
        case 'contributor':
            return $post_status_contributor;
        case 'subscriber':
            return $post_status_subscriber;
        case 'columist':
            return $post_status_columist;
        default:
            return 'pending';
    }
}

/**
 * Helper function to handle file uploads
 * 
 * @param array $files The $_FILES array
 * @return array|int The attachment ID or error array
 */
function kanews_handle_file_upload($files) {
    $attach_id = '';
    
    if (empty($files['featured']['name'])) {
        return $attach_id;
    }
    
    $filename = $files['featured']['name'];
    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    $allowed = array('jpg', 'png', 'jpeg', 'webp');
    
    if (!in_array($ext, $allowed)) {
        return array('error' => esc_html__('Geçersiz dosya formatı!', 'kanews'));
    }
    
    if (is_array($files['featured'])) {
        add_filter('intermediate_image_sizes_advanced', '__return_empty_array');
        $attach_id = upload_user_file($files['featured']);
    }
    
    return $attach_id;
}

/**
 * Helper function to validate security checks
 * 
 * @param array $post_data The $_POST array
 * @return string|bool Error message or false if validation passes
 */
function kanews_validate_security($post_data) {
    if (isset($post_data['g-recaptcha-response']) && $post_data['g-recaptcha-response'] == 0) {
        return '<div class="kanews-response bg-danger">'.esc_html__( 'Güvenlik doğrulamasını gerçekleştirin!', 'kanews' ).'</div>';
    }
    
    if (isset($post_data['cf-turnstile-response']) && empty($post_data['cf-turnstile-response'])) {
        return '<div class="kanews-response bg-danger">'.esc_html__( 'Güvenlik doğrulamasını gerçekleştirin!', 'kanews' ).'</div>';
    }
    
    return false;
}

/**
 * Helper function to send notification emails
 * 
 * @param array $admin_to Admin email addresses
 * @param string $site_name Site name
 * @param int $user_id User ID
 * @param string $title Post title
 * @param string $content Post content
 * @param int $pid Post ID
 * @param string $primary_color Primary color
 * @param string $email_type Type of email (content, company, etc.)
 * @return void
 */
function kanews_send_notification_email($admin_to, $site_name, $user_id, $title, $content, $pid, $primary_color, $email_type = 'content') {
    if (!kanews_get_option('account-logs')) {
        return;
    }
    
    $user_name = get_userdata($user_id)->display_name;
    
    switch ($email_type) {
        case 'content':
            $email_title = sprintf(esc_html__('%s adlı üye bir içerik yayınlandı. - %s', 'kanews'), $user_name, $site_name);
            break;
        case 'company':
            $email_title = sprintf(esc_html__('%s adlı üye bir firma yayınlandı. - %s', 'kanews'), $user_name, $site_name);
            break;
        case 'company_update':
            $email_title = sprintf(esc_html__('%s adlı üye firma bilgilerini güncelledi. - %s', 'kanews'), $user_name, $site_name);
            break;
        case 'content_update':
            $email_title = sprintf(esc_html__('%s adlı yazı düzenlendi. - %s', 'kanews'), $title, $site_name);
            break;
        default:
            $email_title = sprintf(esc_html__('%s adlı üye bir içerik yayınlandı. - %s', 'kanews'), $user_name, $site_name);
    }
    
    kanews_send_mail($admin_to, $email_title, [
        'title' => $email_title,
        'subtitle' => '',
        'body_title' => wp_strip_all_tags($title),
        'body_excerpt' => wp_trim_words(wp_strip_all_tags($content), 55, NULL),
        'button_link' => get_edit_post_link($pid),
        'copyright' => sprintf(esc_html__('Copyright © %s %s Tüm Hakları Saklıdır', 'kanews'), $site_name, date('Y')),
        'color' => $primary_color
    ]);
}

global $wp;
wp_enqueue_script('kanews-ilan-search');
$primary_color = !empty(kanews_get_option('primary-color')) ? kanews_get_option('primary-color') : '#d7050f';
$admin_to[] = get_option( 'admin_email' );
$site_name = get_bloginfo( 'name' );
$user_id = '';

if(is_user_logged_in()) {
	global $current_user, $wp_roles;
	wp_get_current_user();

	$user_id = get_current_user_id();

	$action2 = isset($_GET['islem']) ? $_GET['islem'] : NULL;
	$durum = isset($_GET['durum']) ? $_GET['durum'] : NULL;

	$error = array();  
	if( ('POST' == $_SERVER['REQUEST_METHOD'] && !kanews_user_status($user_id) ) && !isset($_POST['verification_submit']) ) {
		if($action2 != NULL) {
			$error = '<div class="kanews-response bg-danger">'.esc_html__( 'Değişiklikler kaydedilmemiştir lütfen e-posta adresinizi doğrulayın!', 'kanews' ).'</div>';
		}
	} else {
		if ( 'POST' == $_SERVER['REQUEST_METHOD'] && !empty( $_POST['action'] ) && $_POST['action'] == 'update-user' ) {
			if ( count($error) == 0 ) {
				do_action('edit_user_profile_update', $current_user->ID, $_FILES);
				wp_redirect( get_permalink().'?updated=true' ); 
				exit;
			}
		}

		
		
		if($action2 == 'icerik-ekle' || $action2 == 'kose-yazisi-ekle') {

			$posttype = 'post';
			if($action2 == 'kose-yazisi-ekle') {
				add_filter( 'pre_get_document_title', function( $title ) { return esc_html__('Köşe Yazısı Ekle', 'kanews'); }, 10 );
				add_filter( 'wpseo_title', function( $title ) { return esc_html__('Köşe Yazısı ekle', 'kanews'); }, 10 );
				$posttype = 'columist';
			} else {
				add_filter( 'pre_get_document_title', function( $title ) { return esc_html__('İçerik Ekle', 'kanews'); }, 10 );
				add_filter( 'wpseo_title', function( $title ) { return esc_html__('İçerik Ekle', 'kanews'); }, 10 );
			}

			$error = $attach_id = '';
			
			if(isset($_POST['post-submit']) && ( kanews_get_option('post-submit') || kanews_get_option('columnist-submit') )) {
				// Validate form submission
				$title = trim( sanitize_text_field( $_POST[ 'post_title' ] ) );
				$post_excerpt = isset($_POST[ 'post_excerpt' ]) ? wp_strip_all_tags(trim( sanitize_text_field( $_POST[ 'post_excerpt' ] ) )) : '';
				$content = $_POST[ 'content' ];
				$tags = isset($_POST['post_tags']) ? $_POST['post_tags'] : '';

				// Security validation
				$security_error = kanews_validate_security($_POST);
				if ($security_error) {
					$error = $security_error;
				}

				// Content validation
				if(empty($title) || empty($content)) {
					$error = '<div class="kanews-response bg-danger">'.esc_html__( 'Başlık veya içerik boş olmamalı!', 'kanews' ).'</div>';
				}

				// Handle file upload
				if ( ! empty( $_FILES['featured']['name'] ) && empty($error)) {
					$attach_id = kanews_handle_file_upload($_FILES);
					
					if(isset($attach_id['error'])) {
						$error = '<div class="kanews-response bg-danger">'.$attach_id['error'].'</div>';
					}
				}

				// Process post if no errors
				if(empty($error)) {
					// Determine post status based on user role
					$post_status = kanews_get_post_status_by_role($user_id);

					// Prepare post data
					if($action2 == 'kose-yazisi-ekle') {
						$post_info = array(
							'post_title' => wp_strip_all_tags( $title ),
							'post_content' => $content,
							'post_status' => $post_status,
							'post_author' => $user_id,
							'post_type' => 'columist'
						);
					} else {
						$post_info = array(
							'post_title' => wp_strip_all_tags( $title ),
							'post_content' => $content,
							'post_status' => $post_status,
							'post_author' => $user_id,
							'tags_input'    => $tags,
							'post_type' => 'post'
						);
						if( isset($_POST['cat']) ) {
							$post_info['post_category'] = $_POST['cat'];
						}
					}
					
					// Insert post
					$pid = wp_insert_post( $post_info );

					// Update post meta
					if(!empty($post_excerpt)) {
						wp_update_post( array(
							'ID' => $pid,
							'post_excerpt' => wp_strip_all_tags($post_excerpt)
						));
					}

					// Set featured image
					if(!isset($attach_id['error'])) {
						update_post_meta($pid,'_thumbnail_id',$attach_id);
						set_post_thumbnail( $pid, $attach_id );
					}
					
					// Handle success
					if ( !is_wp_error( $pid ) ) {
						$error = '<div class="kanews-response bg-success">'.esc_html__('İçerik gönderildi!', 'kanews').'</div>';
						
						// Send notification email if enabled
						if( kanews_get_option('account-logs') ) {
							kanews_send_notification_email($admin_to, $site_name, $user_id, $title, $content, $pid, $primary_color);
						}
					}
				}
			}
		}elseif($action2 == 'icerik-duzenle') {
			add_filter( 'pre_get_document_title', function( $title ) { return esc_html__('İçerik Düzenle', 'kanews'); }, 10 );
			add_filter( 'wpseo_title', function( $title ) { return esc_html__('İçerik Düzenle', 'kanews'); }, 10 );
			$error = $attach_id = '';
			if(isset($_GET['id']))
			$content = get_post( $_GET['id'] );
			if(isset($_POST['post-edit']) && kanews_get_option('post-edit') && isset($_GET['id']) && is_numeric($_GET['id']) &&  ( 'publish' == get_post_status ( $_GET['id'] ) || 'pending' == get_post_status ( $_GET['id'] ) ) && ($content->post_author == $current_user->ID ) ) {

				$title = trim( sanitize_text_field( $_POST[ 'post_title' ] ) );
				$post_excerpt = trim( sanitize_text_field( $_POST[ 'post_excerpt' ] ) );
				$content = $_POST[ 'content' ];

				$security_error = kanews_validate_security($_POST);
				if ($security_error) {
					$error = $security_error;
				}

				if(empty($error)) {
					$post_info = array(
						'ID' => esc_sql($_GET['id']),
						'post_title' => wp_strip_all_tags( $title ),
						'post_content' => wp_kses_post(wpautop($content)),
						'post_excerpt' => wp_strip_all_tags($post_excerpt),
						'post_category'	=> (!empty($_POST['cat'])) ? $_POST['cat'] : '',
					);

					$user_meta = get_userdata($user_id);

					if($user_meta->roles[0] == 'subscriber')
					$post_info['status'] = 'pending';
					
					$pid = wp_update_post( $post_info, true );
				}

				if ( !is_wp_error( $pid ) ) {
			
		
					if ( ! empty( $_FILES['featured']['name'] )) {
						$filename = $_FILES['featured']['name'];
						$ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
						$allowed = array('jpg', 'png', 'jpeg', 'webp');
						if( in_array( $ext, $allowed ) ){
							foreach( $_FILES as $file ) {
								if( is_array( $file ) ) {
									$attach_id = upload_user_file( $file );
								}
							}
						} else {
							$error = '<div class="kanews-response bg-danger">'.esc_html__('Bir şeyler yanlış gitti!', 'kanews').'</div>';
						}
					}
					
					
					if(!isset($attach_id['error']) && $attach_id != '') {
						update_post_meta($pid,'_thumbnail_id',$attach_id);
						set_post_thumbnail( $pid, $attach_id );
					}
					$error = '<div class="kanews-response bg-success">İçerik başarıyla düzenlenmiştir! <a href="'.get_the_permalink($pid).'">'.esc_html__('Görüntüle', 'kanews').'</a></div>';
					if( kanews_get_option('account-logs') ) {
						kanews_send_notification_email($admin_to, $site_name, $user_id, $title, $content, $pid, $primary_color, 'content_update');
					}
				} else {
					$error = esc_html__('Düzenlenemedi.', 'kanews');
				}
			
			}
		}elseif($action2 == 'firma-ekle') {
			$error = $attach_id = '';
			add_filter( 'pre_get_document_title', function( $title ) { return esc_html__('Firma Ekle', 'kanews'); }, 10 );
			add_filter( 'wpseo_title', function( $title ) { return esc_html__('Firma ekle', 'kanews'); }, 10 );

			if(isset($_POST['firma-submit']) && kanews_get_option('firma-rehberi') ) {
				
				$title = trim( sanitize_text_field( $_POST[ 'comp_name' ] ) );
				$content = $_POST[ 'editor' ];

				$security_error = kanews_validate_security($_POST);
				if ($security_error) {
					$error = $security_error;
				}
				
				if(!is_email($_POST[ 'comp_mail' ] )) 
				$error = '<div class="kanews-response bg-danger">'.esc_html__( 'Lütfen doğru bir e-posta adresi girin!', 'kanews' ).'</div>';

				if(empty($title) && empty($content)) 
				$error = '<div class="kanews-response bg-danger">'.esc_html__( 'Başlık ve içerik boş olmamalı!', 'kanews' ).'</div>';

				if ( ! empty( $_FILES['featured']['name'] ) && empty($error)) {
					$attach_id = kanews_handle_file_upload($_FILES);
					
					if(isset($attach_id['error'])) {
						$error = '<div class="kanews-response bg-danger">'.$attach_id['error'].'</div>';
					}
				}

				if(isset($attach_id['error'])) {
					$error = '<div class="kanews-response bg-danger">'.$attach_id['error'].'</div>';
				}

				$post_status_subscriber = kanews_get_option('post-submit-status') != '' ? kanews_get_option('post-submit-status') : 'pending';
				$post_status_columist = kanews_get_option('post-submit-status-columist') != '' ? kanews_get_option('post-submit-status-columist') : 'pending';
				$post_status_contributor = kanews_get_option('post-submit-status-contributor') != '' ? kanews_get_option('post-submit-status-contributor') : 'pending';
				$post_status_author = kanews_get_option('post-submit-status-author') != '' ? kanews_get_option('post-submit-status-author') : 'pending';
				$post_status_editor = kanews_get_option('post-submit-status-editor') != '' ? kanews_get_option('post-submit-status-editor') : 'pending';

				$user_meta = get_userdata($user_id);

				if($user_meta->roles[0] == 'administrator')
				$post_status = 'publish';
				elseif($user_meta->roles[0] == 'editor')
				$post_status = $post_status_editor;
				elseif($user_meta->roles[0] == 'author')
				$post_status = $post_status_author;
				elseif($user_meta->roles[0] == 'contributor')
				$post_status = $post_status_contributor;
				elseif($user_meta->roles[0] == 'subscriber')
				$post_status = $post_status_subscriber;
				elseif($user_meta->roles[0] == 'columist')
				$post_status = $post_status_columist;
				else
				$post_status = 'pending';
				
				if(empty($error)) {
					$post_info = array(
						'post_type' => 'firma',
						'post_title' => wp_strip_all_tags( $title ),
						'post_content' => $content,
						'post_status' => $post_status,
						'post_author' => $user_id,
						'meta_input'   => array(
							'comp_name' => wp_strip_all_tags( $title ),
							'comp_auth' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'comp_auth' ] ) )),
							'comp_city' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'comp_city' ] ) )),
							'comp_adress' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'comp_adress' ] ) )),
							'comp_tel' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'comp_tel' ] ) )),
							'comp_fax' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'comp_fax' ] ) )),
							'comp_map' => esc_url_raw( $_POST[ 'comp_map' ] ),
							'comp_site' => esc_url_raw( $_POST[ 'comp_site' ] ),
							'comp_mail' => wp_strip_all_tags( trim( sanitize_email( $_POST[ 'comp_mail' ] ) )),
						),
					);


					$pid = wp_insert_post( $post_info );

					if ( !is_wp_error( $pid ) ) {

						$cats_array = array();
						if( isset( $_POST['cat'] ) ) {
							foreach( $_POST['cat'] as $cat ) {
								$cats_array[] = get_term_by( 'id', $cat, 'kategori')->name;
							}
							wp_set_object_terms($pid, $cats_array, 'kategori');
						}

						if(!isset($attach_id['error'])) {
							update_post_meta($pid,'_thumbnail_id',$attach_id);
							set_post_thumbnail( $pid, $attach_id );
						}
						$error = '<div class="kanews-response bg-success">'.esc_html__( 'Firma eklendi!', 'kanews' ).'</div>';
						if( kanews_get_option('account-logs') ) {
							kanews_send_notification_email($admin_to, $site_name, $user_id, $title, $content, $pid, $primary_color, 'company');
						}
					}
				}
			}

		}elseif($action2 == 'firma-duzenle') {
			$error = $attach_id = '';

			if(isset($_GET['id']))
			$content = get_post( $_GET['id'] );
			add_filter( 'pre_get_document_title', function( $title ) { return esc_html__('Firma Düzenle', 'kanews'); }, 10 );
			add_filter( 'wpseo_title', function( $title ) { return esc_html__('Firma Düzenle', 'kanews'); }, 10 );
			if(isset($_GET['r'])) {
				if(kanews_get_option('firma-rehberi') && kanews_get_option('firma-edit')  && isset($_GET['id']) && is_numeric($_GET['id']) &&  ( 'publish' == get_post_status ( $_GET['id'] ) || 'pending' == get_post_status ( $_GET['id'] ) ) && ($content->post_author == $current_user->ID ) ) {
					$post_info = array(
						'ID' => esc_sql($_GET['id']),
						'post_status'   => 'draft',
					);
					$pid = wp_update_post( $post_info, true );
				}
			} else {
				if(isset($_POST['firma-edit']) && kanews_get_option('firma-rehberi') && kanews_get_option('firma-edit')  && isset($_GET['id']) && is_numeric($_GET['id']) &&  ( 'publish' == get_post_status ( $_GET['id'] ) || 'pending' == get_post_status ( $_GET['id'] ) ) && ($content->post_author == $current_user->ID ) ) {
					
					$title = trim( sanitize_text_field( $_POST[ 'comp_name' ] ) );
					$content = $_POST[ 'editor' ];

					$security_error = kanews_validate_security($_POST);
					if ($security_error) {
						$error = $security_error;
					}
					
					if(!is_email($_POST[ 'comp_mail' ] )) 
					$error = '<div class="kanews-response bg-danger">'.esc_html__( 'Lütfen doğru bir e-posta adresi girin!', 'kanews' ).'</div>';

					if(empty($title) && empty($content)) 
					$error = '<div class="kanews-response bg-danger">'.esc_html__( 'Başlık ve içerik boş olmamalı!', 'kanews' ).'</div>';

					if ( ! empty( $_FILES['featured']['name'] ) && empty($error)) {
						$attach_id = kanews_handle_file_upload($_FILES);
						
						if(isset($attach_id['error'])) {
							$error = '<div class="kanews-response bg-danger">'.$attach_id['error'].'</div>';
						}
					}

					if(isset($attach_id['error'])) {
						$error = '<div class="kanews-response bg-danger">'.$attach_id['error'].'</div>';
					}

					if(empty($error)) {
						$post_info = array(
							'post_type' => 'firma',
							'ID' => esc_sql($_GET['id']),
							'post_title' => wp_strip_all_tags( $title ),
							'post_content' => $content,
							'post_author' => $user_id,
							'meta_input'   => array(
								'comp_name' => wp_strip_all_tags( $title ),
								'comp_auth' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'comp_auth' ] ) )),
								'comp_city' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'comp_city' ] ) )),
								'comp_adress' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'comp_adress' ] ) )),
								'comp_tel' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'comp_tel' ] ) )),
								'comp_fax' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'comp_fax' ] ) )),
								'comp_map' => esc_url_raw( $_POST[ 'comp_map' ] ),
								'comp_site' => esc_url_raw( $_POST[ 'comp_site' ] ),
								'comp_mail' => wp_strip_all_tags( trim( sanitize_email( $_POST[ 'comp_mail' ] ) )),
							),
						);

						if(isset($_POST['cat']))
						$post_info['tax_input'] = array('kategori' => $_POST['cat']);

						$user_meta = get_userdata($user_id);
						if($user_meta->roles[0] == 'subscriber')
						$post_info['status'] = 'pending';

						$pid = wp_update_post( $post_info );

						if ( !is_wp_error( $pid ) ) {
							if(!isset($attach_id['error']) && ! empty( $_FILES['featured']['name'] )) {
								update_post_meta($pid,'_thumbnail_id',$attach_id);
								set_post_thumbnail( $pid, $attach_id );
							}
							$error = '<div class="kanews-response bg-success">'.esc_html__( 'Firma düzenlendi!', 'kanews' ).'</div>';
							if( kanews_get_option('account-logs') ) {
								kanews_send_notification_email($admin_to, $site_name, $user_id, $title, $content, $pid, $primary_color, 'company_update');
							}
						} else {
							$error = esc_html__('Düzenlenemedi.', 'kanews');
						}
					}
				}
			}

		}elseif($action2 == 'ilan-ekle') {
			$error = $attach_id = '';
			add_filter( 'pre_get_document_title', function( $title ) { return esc_html__('İlan Ekle', 'kanews'); }, 10 );
			add_filter( 'wpseo_title', function( $title ) { return esc_html__('İlan Ekle', 'kanews'); }, 10 );

			if(isset($_POST['ilan-submit']) && kanews_get_option('ilan-sistemi') && kanews_get_option('ilan-submit') ) {
				
				$title = trim( sanitize_text_field( $_POST[ 'ilan_name' ] ) );
				$content = $_POST[ 'editor' ];

				$security_error = kanews_validate_security($_POST);
				if ($security_error) {
					$error = $security_error;
				}

				if(empty($title) && empty($content)) 
				$error = '<div class="kanews-response bg-danger">'.esc_html__( 'Başlık ve içerik boş olmamalı!', 'kanews' ).'</div>';;

				if( (empty($_POST[ 'il' ]) || (isset($_POST[ 'il' ]) && $_POST[ 'il' ] < 1)) && empty($_POST[ 'ulke' ]) ) 
				$error = '<div class="kanews-response bg-danger">'.esc_html__( 'İl Seç', 'kanews' ).'</div>';

				if ( ! empty( $_FILES['featured']['name'] ) && empty($error)) {
					$attach_id = kanews_handle_file_upload($_FILES);
					
					if(isset($attach_id['error'])) {
						$error = '<div class="kanews-response bg-danger">'.$attach_id['error'].'</div>';
					}
				}

				$post_status_subscriber = kanews_get_option('post-submit-status') != '' ? kanews_get_option('post-submit-status') : 'pending';
				$post_status_columist = kanews_get_option('post-submit-status-columist') != '' ? kanews_get_option('post-submit-status-columist') : 'pending';
				$post_status_contributor = kanews_get_option('post-submit-status-contributor') != '' ? kanews_get_option('post-submit-status-contributor') : 'pending';
				$post_status_author = kanews_get_option('post-submit-status-author') != '' ? kanews_get_option('post-submit-status-author') : 'pending';
				$post_status_editor = kanews_get_option('post-submit-status-editor') != '' ? kanews_get_option('post-submit-status-editor') : 'pending';

				$user_meta = get_userdata($user_id);

				if($user_meta->roles[0] == 'administrator')
				$post_status = 'publish';
				elseif($user_meta->roles[0] == 'editor')
				$post_status = $post_status_editor;
				elseif($user_meta->roles[0] == 'author')
				$post_status = $post_status_author;
				elseif($user_meta->roles[0] == 'contributor')
				$post_status = $post_status_contributor;
				elseif($user_meta->roles[0] == 'subscriber')
				$post_status = $post_status_subscriber;
				elseif($user_meta->roles[0] == 'columist')
				$post_status = $post_status_columist;
				else
				$post_status = 'pending';

				$ilan_list = isset( $_POST['ilan_list'] ) ? (array) $_POST['ilan_list'] : array();

				if(empty($error)) {
					$post_info = array(
						'post_type'   => 'ilan',
						'post_title'  => wp_strip_all_tags( $title ),
						'post_content'=> $content,
						'post_status' => $post_status,
						'post_author' => $user_id,
						'meta_input'  => array(
							'ilan_name' => wp_strip_all_tags( $title ),
							'ilan_price' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'ilan_price' ] ) )),
							'ilan_mail' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'ilan_mail' ] ) )),
							'ilan_tel'  => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'ilan_tel' ] ) )),
							'ilan_tel2' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'ilan_tel2' ] ) )),
							'ilan_adress' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'ilan_adress' ] ) )),
							'ilan_whatsapp' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'ilan_whatsapp' ] ) )),
							'ilan_map'  => esc_url_raw( $_POST[ 'ilan_map' ] ),
							'ilan_list'  =>  $ilan_list,
						)
					);

					$pid = wp_insert_post( $post_info );
					

					if ( !is_wp_error( $pid ) ) {
						$cats_array = array();
						if( isset( $_POST['cat'] ) ) {
							foreach( $_POST['cat'] as $cat ) {
								$cats_array[] = get_term_by( 'id', $cat, 'ilan-kategori')->name;
							}
							wp_set_object_terms($pid, $cats_array, 'ilan-kategori');
						}
						
						if( isset($_POST['post_tags']) )
						wp_set_object_terms($pid,  $_POST['post_tags'], 'ilan-etiket');

						if( isset($_POST[ 'il' ]) && $_POST[ 'il' ] > 0 ) {
							$ilan_city = kanews_get_sehir($_POST[ 'il' ]);
							wp_set_object_terms($pid, array($ilan_city), 'ilan-il');
						} elseif( isset($_POST[ 'ulke' ]) && $_POST[ 'ulke' ] != '' ) {
							$ilan_city = sanitize_text_field($_POST[ 'ulke' ]);
							wp_set_object_terms($pid, array($ilan_city), 'ilan-il');
						}

						if( isset($_POST[ 'ilce' ]) && $_POST[ 'ilce' ] != '' ) {
							$ilan_town = sanitize_text_field($_POST[ 'ilce' ]);
							wp_set_object_terms($pid, array($ilan_town), 'ilan-ilce');
						} elseif( isset($_POST[ 'sehir' ]) && $_POST[ 'sehir' ] != '' ) {
							$ilan_town = sanitize_text_field($_POST[ 'sehir' ]);
							wp_set_object_terms($pid, array($ilan_town), 'ilan-ilce');
						}

						if(!isset($attach_id['error'])) {
							update_post_meta($pid,'_thumbnail_id',$attach_id);
							set_post_thumbnail( $pid, $attach_id );
						}
						for ($x = 1; $x <= 5; $x++) {
							if ( ! empty( $_FILES["ilan_thumb$x"]['name'] )) {
								$filename = $_FILES["ilan_thumb$x"]['name'];
								$ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
								$allowed = array('jpg', 'png', 'jpeg', 'webp');
								if( in_array( $ext, $allowed ) ){

									if( is_array( $_FILES["ilan_thumb$x"] ) ) {
										$attach_id = upload_user_file( $_FILES["ilan_thumb$x"] );
									}
									
								}
								if(!isset($attach_id['error'])) {
									$attach_id = array('id' => $attach_id, 'url' => wp_get_attachment_image_src( $attach_id, 'full')[0], 'width' => wp_get_attachment_image_src( $attach_id, 'full')[1], 'height' => wp_get_attachment_image_src( $attach_id, 'full')[2], 'thumbnail' => wp_get_attachment_image_src( $attach_id, 'thumbnail')[0], 'alt' => 'ilan_thumb'.$x, 'title' => 'ilan_thumb'.$x, 'description' => '' );
									update_post_meta($pid,"ilan_thumb$x",$attach_id);
								}
							}
						}

						$error = '<div class="kanews-response bg-success">'.esc_html__( 'İlan Eklendi!', 'kanews' ).'</div>';

						if( kanews_get_option('account-logs') ) {
							$email_title = sprintf(esc_html__('%s adlı üye bir ilan yayınlandı. - %s', 'kanews'), get_userdata($user_id)->display_name, $site_name);
							kanews_send_mail($admin_to, $email_title,
							[
								'title' => $email_title,
								'subtitle' => '',
								'body_title' => wp_strip_all_tags( $title ),
								'body_excerpt' =>  wp_trim_words( wp_strip_all_tags($content), 55, NULL ),
								'button_link' => get_edit_post_link($pid),
								'copyright' => sprintf(esc_html__( 'Copyright © %s %s Tüm Hakları Saklıdır', 'kanews' ), $site_name, date('Y')),
								'color' => $primary_color
							] );
						}
					}
				}
				
			}

		}elseif($action2 == 'ilan-duzenle') {
			$error = $attach_id = '';

			if(isset($_GET['id']))
			$content = get_post( $_GET['id'] );
			add_filter( 'pre_get_document_title', function( $title ) { return esc_html__('İlan Düzenle', 'kanews'); }, 10 );
			add_filter( 'wpseo_title', function( $title ) { return esc_html__('İlan Düzenle', 'kanews'); }, 10 );
			if(isset($_GET['r'])) {
				if(kanews_get_option('ilan-sistemi') && kanews_get_option('ilan-edit')  && isset($_GET['id']) && is_numeric($_GET['id']) &&  ( 'publish' == get_post_status ( $_GET['id'] ) || 'pending' == get_post_status ( $_GET['id'] ) ) && ($content->post_author == $current_user->ID ) ) {
					$post_info = array(
						'ID' => esc_sql($_GET['id']),
						'post_status'   => 'draft',
					);
					$pid = wp_update_post( $post_info, true );
				}
			} else {
				if(isset($_POST['ilan-edit']) && kanews_get_option('ilan-sistemi') && kanews_get_option('ilan-edit')  && isset($_GET['id']) && is_numeric($_GET['id']) &&  ( 'publish' == get_post_status ( $_GET['id'] ) || 'pending' == get_post_status ( $_GET['id'] ) ) && ($content->post_author == $current_user->ID ) ) {

					$title = trim( sanitize_text_field( $_POST[ 'ilan_name' ] ) );
					$content = $_POST[ 'editor' ];

					$security_error = kanews_validate_security($_POST);
					if ($security_error) {
						$error = $security_error;
					}

					if(empty($title) && empty($content)) 
					$error = '<div class="kanews-response bg-danger">'.esc_html__( 'Başlık ve içerik boş olmamalı!', 'kanews' ).'</div>';;

					if( (empty($_POST[ 'il' ]) || (isset($_POST[ 'il' ]) && $_POST[ 'il' ] < 1)) && empty($_POST[ 'ulke' ]) ) 
					$error = '<div class="kanews-response bg-danger">'.esc_html__( 'İl Seç', 'kanews' ).'</div>';

					if ( ! empty( $_FILES['featured']['name'] ) && empty($error)) {
						$attach_id = kanews_handle_file_upload($_FILES);
						
						if(isset($attach_id['error'])) {
							$error = '<div class="kanews-response bg-danger">'.$attach_id['error'].'</div>';
						}
					}

					if(isset($attach_id['error'])) {
						$error = '<div class="kanews-response bg-danger">'.$attach_id['error'].'</div>';
					}

					$ilan_list = isset( $_POST['ilan_list'] ) ? (array) $_POST['ilan_list'] : array();


					if(empty($error)) {
						$post_info = array(
							'ID' => esc_sql($_GET['id']),
							'post_title'  => wp_strip_all_tags( $title ),
							'post_content'=> $content,
							'meta_input'  => array(
								'ilan_name' => wp_strip_all_tags( $title ),
								'ilan_price' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'ilan_price' ] ) )),
								'ilan_mail' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'ilan_mail' ] ) )),
								'ilan_tel'  => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'ilan_tel' ] ) )),
								'ilan_tel2' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'ilan_tel2' ] ) )),
								'ilan_adress' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'ilan_adress' ] ) )),
								'ilan_whatsapp' => wp_strip_all_tags( trim( sanitize_text_field( $_POST[ 'ilan_whatsapp' ] ) )),
								'ilan_map'  => esc_url_raw( $_POST[ 'ilan_map' ] ),
								'ilan_list'  =>  $ilan_list,
							)
						);

						$user_meta = get_userdata($user_id);
						if($user_meta->roles[0] == 'subscriber')
						$post_info['status'] = 'pending';

						$pid = wp_update_post( $post_info, true );

						if ( !is_wp_error( $pid ) ) {

							$cats_array = array();
							if( isset( $_POST['cat'] ) ) {
								foreach( $_POST['cat'] as $cat ) {
									$cats_array[] = get_term_by( 'id', $cat, 'ilan-kategori')->name;
								}
								wp_set_object_terms($pid, $cats_array, 'ilan-kategori');
							}
							
							if( isset($_POST['post_tags']) )
							wp_set_object_terms($pid,  $_POST['post_tags'], 'ilan-etiket');

							if( isset($_POST[ 'il' ]) && $_POST[ 'il' ] > 0 ) {
								$ilan_city = kanews_get_sehir($_POST[ 'il' ]);
								wp_set_object_terms($pid, array($ilan_city), 'ilan-il');
							} elseif( isset($_POST[ 'ulke' ]) && $_POST[ 'ulke' ] != '' ) {
								$ilan_city = sanitize_text_field($_POST[ 'ulke' ]);
								wp_set_object_terms($pid, array($ilan_city), 'ilan-il');
							}

							if( isset($_POST[ 'ilce' ]) && $_POST[ 'ilce' ] != '' ) {
								$ilan_town = sanitize_text_field($_POST[ 'ilce' ]);
								wp_set_object_terms($pid, array($ilan_town), 'ilan-ilce');
							} elseif( isset($_POST[ 'sehir' ]) && $_POST[ 'sehir' ] != '' ) {
								$ilan_town = sanitize_text_field($_POST[ 'sehir' ]);
								wp_set_object_terms($pid, array($ilan_town), 'ilan-ilce');
							}

							if(!isset($attach_id['error']) && $attach_id != '') {
								update_post_meta($pid,'_thumbnail_id',$attach_id);
								set_post_thumbnail( $pid, $attach_id );
							}
							for ($x = 1; $x <= 5; $x++) {
								if ( ! empty( $_FILES["ilan_thumb$x"]['name'] )) {
									$filename = $_FILES["ilan_thumb$x"]['name'];
									$ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
									$allowed = array('jpg', 'png', 'jpeg', 'webp');
									if( in_array( $ext, $allowed ) ){

										if( is_array( $_FILES["ilan_thumb$x"] ) ) {
											$attach_id = upload_user_file( $_FILES["ilan_thumb$x"] );
										}
										
									}
									if(!isset($attach_id['error']) && !$attach_id != '') {
										$attach_id = array('id' => $attach_id, 'url' => wp_get_attachment_image_src( $attach_id, 'full')[0], 'width' => wp_get_attachment_image_src( $attach_id, 'full')[1], 'height' => wp_get_attachment_image_src( $attach_id, 'full')[2], 'thumbnail' => wp_get_attachment_image_src( $attach_id, 'thumbnail')[0], 'alt' => 'ilan_thumb'.$x, 'title' => 'ilan_thumb'.$x, 'description' => '' );
										update_post_meta($pid,"ilan_thumb$x",$attach_id);
									}
								}
							}

							$error = '<div class="kanews-response bg-success">'.esc_html__('İlan başarıyla düzenlenmiştir!', 'kanews').' <a href="'.get_the_permalink($pid).'">'.esc_html__('Görüntüle', 'kanews').'</a></div>';
							if( kanews_get_option('account-logs') ) {
								$email_title = sprintf(esc_html__('%s isimli ilan güncellendi. - %s', 'kanews'), $title, $site_name);
								kanews_send_mail($admin_to, $email_title,
								[
									'title' => $email_title,
									'subtitle' => '',
									'body_title' => wp_strip_all_tags( $title ),
									'body_excerpt' =>  wp_trim_words( wp_strip_all_tags($content), 55, NULL ),
									'button_link' => get_edit_post_link($pid),
									'copyright' => sprintf(esc_html__( 'Copyright © %s %s Tüm Hakları Saklıdır', 'kanews' ), $site_name, date('Y')),
									'color' => $primary_color
								] );
							}
						} else {
							$error = esc_html__('Düzenlenemedi.', 'kanews');
						}
					}
					
				}
			}

		}elseif($action2 == 'takip') {
			add_filter( 'pre_get_document_title', function( $title ) { return esc_html__('Takip Ettiklerim', 'kanews'); }, 10 );
			add_filter( 'wpseo_title', function( $title ) { return esc_html__('Takip Ettiklerim', 'kanews'); }, 10 );
			if($_SERVER['REQUEST_METHOD'] == 'POST') {
				if(isset($_POST['switch'])) 
				update_user_option( $current_user->ID, 'kanews_mail_notification', 'true' );
				else
				update_user_option( $current_user->ID, 'kanews_mail_notification', 'false' );
			}
		}elseif($action2 == 'ayarlar') {
			add_filter( 'pre_get_document_title', function( $title ) { return esc_html__('Hesap Ayarları', 'kanews'); }, 10 );
			add_filter( 'wpseo_title', function( $title ) { return esc_html__('Hesap Ayarları', 'kanews'); }, 10 );
		}elseif($action2 == 'iceriklerim') {
			add_filter( 'pre_get_document_title', function( $title ) { return esc_html__('İçeriklerim', 'kanews'); }, 10 );
			add_filter( 'wpseo_title', function( $title ) { return esc_html__('İçeriklerim', 'kanews'); }, 10 );
		}elseif($action2 == 'ilanlarim') {
			add_filter( 'pre_get_document_title', function( $title ) { return esc_html__('İlanlarım', 'kanews'); }, 10 );
			add_filter( 'wpseo_title', function( $title ) { return esc_html__('İlanlarım', 'kanews'); }, 10 );
		}elseif($action2 == 'firmalarim') {
			add_filter( 'pre_get_document_title', function( $title ) { return esc_html__('Firmalarım', 'kanews'); }, 10 );
			add_filter( 'wpseo_title', function( $title ) { return esc_html__('Firmalarım', 'kanews'); }, 10 );
		}elseif($action2 == 'yorumlar') {
			add_filter( 'pre_get_document_title', function( $title ) { return esc_html__('Yorumlarım', 'kanews'); }, 10 );
			add_filter( 'wpseo_title', function( $title ) { return esc_html__('Yorumlarım', 'kanews'); }, 10 );
		}elseif($action2 == 'favoriler') {
			add_filter( 'pre_get_document_title', function( $title ) { return esc_html__('Favorilerim', 'kanews'); }, 10 );
			add_filter( 'wpseo_title', function( $title ) { return esc_html__('Favorilerim', 'kanews'); }, 10 );
		} else {
			if(kanews_get_option('email-activation')) {
				if(isset($_GET['verification']) && ! kanews_user_status($user_id)) {
					$transient_name = 'kanews-user-auth-cache-' . $user_id;
					$transient_time = 15 * MINUTE_IN_SECONDS;
					if ( false === ( $user_auth_code = get_transient( $transient_name ) ) ){
						kanews_send_user_activation_mail($user_id);
					}
				}
				if ( isset($_POST['verification_submit']) && ! kanews_user_status($user_id) ) {
					if( isset($_POST['verification_code']) && !empty($_POST['verification_code']) ) {
						$activation = get_user_meta($user_id, '_kanews_user_activation', true);
						if(is_array($activation) && isset($activation['token'])) {
							if($_POST['verification_code'] == $activation['token']) {
								$timediff = strtotime("now") - $activation['expired'];
								if($timediff > 30 * MINUTE_IN_SECONDS) {
									$error = '<div class="kanews-response bg-warning">'.esc_html__('Girdiğiniz tokenın süresi dolmuştur.', 'kanews').'</div>';
								} else {
									$args = array(
										'status'    => true,
										'update'   => strtotime(date("c"))
									);
									update_user_meta($user_id, '_kanews_user_activation', $args);
									$transient_name = 'kanews-user-auth-cache-' . $user_id;
									delete_transient( $transient_name );
								}
							} else {
								$error = '<div class="kanews-response bg-danger">'.esc_html__('Doğrulama kodunuz hatalı!', 'kanews').'</div>';
							}
						} else {
							$error = '<div class="kanews-response bg-danger">'.esc_html__('Hata oluştu, tekrar deneyiniz!', 'kanews').'</div>';
						}
					}
				}
			}
		}
	}

} else {
	$action2 = isset($_GET['islem']) ? $_GET['islem'] : NULL;
}

if(is_user_logged_in()):
get_header();
else:?>
<!doctype html>
<html <?php language_attributes(); ?>>
<head>
	<meta charset="<?php bloginfo( 'charset' ); ?>">
	<meta http-equiv="X-UA-Compatible" content="IE=edge">
	<meta name="viewport" content="width=device-width, initial-scale=1, maximum-scale=1, viewport-fit=cover">
	<link rel="profile" href="https://gmpg.org/xfn/11">
	<?php wp_head(); ?>
</head>
<body <?php body_class('css-transitions-only-after-page-load'); ?>>
<div id="page" class="site-wrapper">
<?php wp_body_open(); ?>
<?php endif;?>
<?php if ( have_posts() ) : while ( have_posts() ) : the_post(); ?>
	<?php if(is_user_logged_in()): ?>
		<div class="kanews-page-header kanews-page-header-special">
			<div class="container">
				<div class="row">
					<div class="col"><h1 class="kanews-page-title"><?php esc_html_e( 'Hesabım', 'kanews' ); ?></h1></div>
				</div>
			</div>
		</div>

		<main id="main" class="site-main">
			<div class="container">
				<div class="row">

					<div class="col-12 col-md-4 mobile-padding">
						<div class="kanews-panel kanews-collapse-wrapper">
							<div class="kanews-panel-top">
								<div class="d-flex align-items-center">
									<div class="col-auto">
										<div style="cursor:pointer" data-toggle="kanews-upload-user-avatar"class="kanews-panel-user-avatar"> 
											<?php echo kanews_user_avatar(get_the_author_meta('ID', $current_user->ID),'90'); ?>
											
										</div>
									</div>
									<div class="col">
										<a href="<?php echo get_author_posts_url( get_the_author_meta( 'ID', $current_user->ID ) ); ?>"><?php echo esc_html(get_the_author_meta('display_name', $current_user->ID)); ?> <?php echo kanews_user_subscribe_badge($current_user->ID); ?></a>
										<?php echo user_badge($current_user->ID); ?>
										<?php $mail = get_the_author_meta('user_email', $current_user->ID); if($mail): ?><div style="word-break: break-word;"><?php echo esc_html($mail); ?></div><?php endif; ?>
									</div>
								</div>
							</div>
							<?php
							$favorilerim = $notifications = $begendiklerim = $kose = $ilan = $yorumlar = $abonelik =  $ayarlar = $main = '';$icerik = ''; $firmalarim = ''; $ilanlarim = ''; $iceriklerim = ''; $firma = ''; $takip = '';
							switch ($action2) {
								case 'favoriler':
									$favorilerim = 'is-active';
								break;
								case 'begeniler':
									$begendiklerim = 'is-active';
								break;
								case 'yorumlar':
									$yorumlar = 'is-active';
								break;
								case 'abonelik':
									$abonelik = 'is-active';
								break;
								case 'ayarlar':
									$ayarlar = 'is-active';
								break;
								case 'iceriklerim':
									$iceriklerim = 'is-active';
								break;
								case 'ilanlarim':
									$ilanlarim = 'is-active';
								break;
								case 'ilan-duzenle':
									$ilanlarim = 'is-active';
								break;
								case 'firma-duzenle':
									$firmalarim = 'is-active';
								break;
								case 'firmalarim':
									$firmalarim = 'is-active';
								break;
								case 'icerik-duzenle':
									$iceriklerim = 'is-active';
								break;
								case 'icerik-ekle':
									$icerik = 'is-active';
								break;
								case 'kose-yazisi-ekle':
									$kose = 'is-active';
								break;
								case 'firma-ekle':
									$firma = 'is-active';
								break;
								case 'ilan-ekle':
									$ilan = 'is-active';
								break;
								case 'takip':
									$takip = 'is-active';
								break;
								case 'bildirimler':
									$notifications = 'is-active';
								break;
								default:
									$main = 'is-active';
							}
							?>
							<div class="kanews-collapse-btn hidden-desktop"><i class="icon-arrow icon-3x"></i> <span><?php esc_html_e( 'Daha Fazla Göster', 'kanews' ); ?></span></div>
							<div class="kanews-panel-nav kanews-collapse-item hidden-mobile">
								<ul>
									<li class="bks1"><?php esc_html_e( 'Genel', 'kanews' ); ?></li>
									<li class="<?php echo $main;?>"><a href="<?php the_permalink(); ?>"><i class="icon-grid"></i>  <?php esc_html_e( 'Pano', 'kanews' ); ?></a></li>
									<?php if(kanews_get_option('feed-page') != '')
									echo '<li><a href="'.get_page_link(kanews_get_option('feed-page')).'"><i class="icon-rss"></i> '.esc_html__( 'Akışım', 'kanews' ).'</a></li>'; ?>
									<?php if ( class_exists( 'WooCommerce') && kanews_get_option('kanews-insights') ): ?>
									<li class="<?php echo $abonelik;?>"><a href="<?php echo add_query_arg( array('islem' => 'abonelik'), get_permalink()); ?>"><i class="icon-subscriptions"></i>  <?php esc_html_e( 'Abonelik', 'kanews' ); ?></a></li>
									<?php endif; ?>
									<li class="bks1"><?php esc_html_e( 'Kütüphanem', 'kanews' ); ?></li>
									<?php if( kanews_get_option('follow-categories') ): ?>
									<li class="<?php echo $takip;?>"><a href="<?php echo add_query_arg( array('islem' => 'takip'), get_permalink()); ?>"><i class="icon-add"></i>  <?php esc_html_e( 'Takip Ettiklerim', 'kanews' ); ?></a></li>
									<?php endif; ?>
									<?php if(kanews_get_option('post-like') == true): ?>
										<li class="<?php echo $begendiklerim;?>"><a href="<?php echo add_query_arg( array('islem' => 'begeniler'), get_permalink()); ?>"><i class="icon-like-o"></i>  <?php esc_html_e( 'Beğendiklerim', 'kanews' ); ?></a></li>
									<?php endif; ?>
									<?php if(kanews_get_option('post-bookmark') == true): ?>
									<li class="<?php echo $favorilerim;?>"><a href="<?php echo add_query_arg( array('islem' => 'favoriler'), get_permalink()); ?>"><i class="icon-bookmark-o"></i>  <?php esc_html_e( 'Favorilerim', 'kanews' ); ?></a></li>
									<?php endif; ?>
									<li class="<?php echo $yorumlar;?>"><a href="<?php echo add_query_arg( array('islem' => 'yorumlar'), get_permalink()); ?>"><i class="icon-comment"></i> <?php esc_html_e( 'Yorumlarım', 'kanews' ); ?></a></li>
									<li class="<?php echo $iceriklerim;?>"><a href="<?php echo add_query_arg( array('islem' => 'iceriklerim'), get_permalink()); ?>"><i class="icon-article"></i> <?php esc_html_e( 'İçeriklerim', 'kanews' ); ?></a></li>
									<?php if(kanews_get_option('ilan-sistemi') == true): ?>
									<li class="<?php echo $ilanlarim;?>"><a href="<?php echo add_query_arg( array('islem' => 'ilanlarim'), get_permalink()); ?>"><i class="icon-star-half "></i> <?php esc_html_e( 'İlanlarım', 'kanews' ); ?></a></li>
									<?php endif; ?>
									<?php if(kanews_get_option('firma-rehberi') == true): ?>
									<li class="<?php echo $firmalarim;?>"><a href="<?php echo add_query_arg( array('islem' => 'firmalarim'), get_permalink()); ?>"><i class="icon-local"></i> <?php esc_html_e( 'Firmalarım', 'kanews' ); ?></a></li>
									<?php endif; ?>
									<?php if(kanews_get_option('notifications')): ?>
									<li class="<?php echo $notifications;?>"><a href="<?php echo add_query_arg( array('islem' => 'bildirimler'), get_permalink()); ?>"><i class="icon-notifications"></i>  <?php esc_html_e( 'Bildirimlerim', 'kanews' ); ?></a></li>
									<?php endif; ?>
									<?php if( kanews_get_option('ilan-sistemi') || kanews_get_option('firma-rehberi') || kanews_get_option('post-submit')): ?>
										<li class="bks1"><?php esc_html_e( 'Sistemsel', 'kanews' ); ?></li>
										<?php if(kanews_get_option('post-submit') == true): ?>
										<li class="<?php echo $icerik;?>"><a href="<?php echo add_query_arg( array('islem' => 'icerik-ekle'), get_permalink()); ?>"><i class="icon-edit"></i> <?php esc_html_e( 'İçerik Ekle', 'kanews' ); ?></a></li>
										<?php endif; ?>
										<?php if(kanews_get_option('columnist-submit')): ?>
										<li class="<?php echo $kose;?>"><a href="<?php echo add_query_arg( array('islem' => 'kose-yazisi-ekle'), get_permalink()); ?>"><i class="icon-edit"></i> <?php esc_html_e( 'Köşe Yazısı Ekle', 'kanews' ); ?></a></li>
										<?php endif; ?>
										<?php if(kanews_get_option('firma-rehberi') == true): ?>
										<li class="<?php echo $firma;?>"><a href="<?php echo add_query_arg( array('islem' => 'firma-ekle'), get_permalink()); ?>"><i class="icon-local"></i> <?php esc_html_e( 'Firma Ekle', 'kanews' ); ?></a></li>
										<?php endif; ?>
										<?php if(kanews_get_option('ilan-sistemi') && kanews_get_option('ilan-submit')): ?>
										<li class="<?php echo $ilan;?>"><a href="<?php echo add_query_arg( array('islem' => 'ilan-ekle'), get_permalink()); ?>"><i class="icon-local"></i> <?php esc_html_e( 'İlan Ekle', 'kanews' ); ?></a></li>
										<?php endif; ?>
									<?php endif; ?>
									<li class="bks1"><?php esc_html_e( 'Hesap', 'kanews' ); ?></li>
									<li class="<?php echo $ayarlar;?>"><a href="<?php echo add_query_arg( array('islem' => 'ayarlar'), get_permalink()); ?>"><i class="icon-settings"></i>  <?php esc_html_e( 'Hesap Ayarlarım', 'kanews' ); ?></a></li>
									<li><a style="cursor:pointer" data-toggle="kanews-upload-user-avatar"><i class="icon-gallery"></i>  <?php esc_html_e( 'Profil Görselleri', 'kanews' ); ?></a></li>
									<li><a href="<?php echo wp_logout_url( home_url( add_query_arg( [], $GLOBALS['wp']->request ) ) ); ?>"><i class="icon-logout"></i>  <?php esc_html_e( 'Çıkış Yap', 'kanews' ); ?></a></li>
								</ul>
							</div>
						</div>
					</div>

					<div class="col-12 col-md-8 mobile-padding">
							
						<div class="kanews-custom-page">
				
							<?php switch($action2) {
								case 'takip': 
									if ( kanews_get_option('follow-categories') ):
										get_template_part( 'inc/dashboard/dashboard', 'follow');
									else: 
										echo esc_html_e( 'Bu sayfa geçici olarak devre dışı bırakılmıştır.', 'kanews' ); 
									endif;
								break;
								case 'abonelik': 
									get_template_part( 'inc/dashboard/dashboard', 'subscription');
								break;
								case 'favoriler':
									get_template_part( 'inc/dashboard/dashboard', 'bookmark');
									break;
								case 'begeniler':
									get_template_part( 'inc/dashboard/dashboard', 'like');
								break;
								case 'yorumlar':
									get_template_part( 'inc/dashboard/dashboard', 'comments');
								break;
								case 'iceriklerim':
									get_template_part( 'inc/dashboard/dashboard', 'posts');
								break;
								case 'ilanlarim':
									if(kanews_get_option('ilan-sistemi'))
									get_template_part( 'inc/dashboard/dashboard', 'ilan');
								break;
								case 'firmalarim':
									if(kanews_get_option('firma-rehberi'))
									get_template_part( 'inc/dashboard/dashboard', 'firma');
								break;
								case 'ayarlar':
									get_template_part( 'inc/dashboard/dashboard', 'settings');
									break;
								case 'bildirimler':
									get_template_part( 'inc/dashboard/dashboard', 'notifications');
								break;
								case 'ilan-ekle': if( kanews_get_option('ilan-sistemi') && kanews_get_option('ilan-submit') ): ?>
									<h2 class="kanews-custom-page-title"><?php esc_html_e( 'İlan Ekle', 'kanews' ); ?></h2>
									<p class="mb-3"><?php echo kanews_translate('kanews-add-post-text', esc_html_e( 'İlanınız editörümüz tarafından incelendikten sonra uygun görüldüğü takdirde yayınlanacaktır.', 'kanews' )); ?></p>
									<form action="<?php the_permalink(); ?>?islem=ilan-ekle" method="post" enctype="multipart/form-data">
										<?php echo $error; ?>
  									<div class="row">
											<div class="col-12">
												<div class="kanews-form">      
													<input type="text" id="ilan_name" name="ilan_name" value="<?php echo isset($_POST['ilan_name']) ? $_POST['ilan_name']  : ''; ?>" required autofocus>
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('İlan Adı', 'kanews'); ?> <span class="color-danger">*</span></label>
												</div>
											</div>
											<div class="col-12 col-md-6">
												<div class="kanews-form2 bgw position-relative">
													<input type="text" 
														   id="dashboard-ilan-il-search" 
														   class="mb-0" 
														   placeholder="<?php esc_html_e( 'Ara', 'kanews' ); ?>" 
														   autocomplete="off" />
													<button type="button" class="ilan-clear-btn" id="dashboard-ilan-il-clear" style="display: none;">
														<i class="icon-close"></i>
													</button>
													<select id="dashboard-ilan-il" style="display: none;">
														<option value=""><?php esc_html_e( 'İl Seç', 'kanews' ); ?></option>
													</select>
													<input type="hidden" name="il" id="dashboard-ilan-il-hidden" value="" />
													<div id="dashboard-ilan-il-dropdown" class="ilan-dropdown" style="display: none;"></div>
												</div>
											</div>
											<div class="col-12 col-md-6">
												<div class="kanews-form2 bgw position-relative">
													<input type="text" 
														   id="dashboard-ilan-ilce-search" 
														   class="mb-0" 
														   placeholder="<?php esc_html_e( 'Ara', 'kanews' ); ?>" 
														   autocomplete="off"
														   disabled />
													<button type="button" class="ilan-clear-btn" id="dashboard-ilan-ilce-clear" style="display: none;">
														<i class="icon-close"></i>
													</button>
													<select id="dashboard-ilan-ilce" style="display: none;">
														<option value=""><?php esc_html_e( 'İlçe Seç', 'kanews' ); ?></option>
													</select>
													<input type="hidden" name="ilce" id="dashboard-ilan-ilce-hidden" value="" />
													<div id="dashboard-ilan-ilce-dropdown" class="ilan-dropdown" style="display: none;"></div>
												</div>
											</div>
											
											
											<div class="col-12">
												<div class="kanews-form">      
													<input type="text" id="ilan_adress" name="ilan_adress" value="<?php isset($_POST['ilan_adress']) ? $_POST['ilan_adress']  : ''; ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('Tam Adres', 'kanews'); ?></label>
												</div>
											</div>

											<div class="col-12" hidden>
												<div class="kanews-form">      
													<input value="" type="text" id="ilan_map" name="ilan_map" value="<?php isset($_POST['ilan_map']) ? $_POST['ilan_map']  : ''; ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label ><?php _e('Harita Embed Linki', 'kanews'); ?></label>
												</div>
											</div>
											

											<div class="col-12">
												<div class="kanews-form">      
													<input type="text" id="ilan_price" name="ilan_price" value="<?php echo isset($_POST['ilan_price']) ? $_POST['ilan_price']  : ''; ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('İlan Fiyatı', 'kanews'); ?></label>
												</div>
											</div>
											<div class="col-12 col-md-6">
												<div class="kanews-form">      
													<input type="text" id="ilan_tel" name="ilan_tel" value="<?php echo isset($_POST['ilan_tel']) ? $_POST['ilan_tel']  : ''; ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('Telefon Numarası', 'kanews'); ?></label>
												</div>
											</div>
											<div class="col-12 col-md-6">
												<div class="kanews-form">      
													<input type="text" id="ilan_tel2" name="ilan_tel2" value="<?php echo isset($_POST['ilan_tel2']) ? $_POST['ilan_tel2']  : ''; ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('Telefon Numarası', 'kanews'); ?> 2</label>
												</div>
											</div>
                                            <div class="col-12 col-md-6">
                                                <div class="kanews-form">      
                                                    <input type="text" id="ilan_whatsapp" name="ilan_whatsapp" value="<?php echo isset($_POST['ilan_whatsapp']) ? $_POST['ilan_whatsapp']  : ''; ?>">
                                                    <span class="highlight"></span>
                                                    <span class="bar"></span>
                                                    <label><?php _e('Whatsapp Numarası', 'kanews'); ?></label>
                                                    <small class="text-muted" style="display:block;margin-top:4px;"><?php esc_html_e('wa.me’den sonrasını yazın, örn: 905551112233', 'kanews'); ?></small>
                                                </div>
                                            </div>
											<div class="col-12 col-md-6">
												<div class="kanews-form">      
													<input type="email" id="ilan_mail" name="ilan_mail" value="<?php echo isset($_POST['ilan_mail'])  ? $_POST['ilan_mail']  : ''; ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('E-Posta', 'kanews'); ?></label>
												</div>
											</div>

											<div class="col-12 mb-1">
												<div class="kanews-form">
													<?php wp_dropdown_categories( 'show_option_none='.__('Kategori Seç', 'kanews').'&tab_index=4&taxonomy=ilan-kategori&multiple=1&hide_empty=0' ); ?>
												</div>
											</div>

											<div class="col-12">
												<div class="kanews-form">      
													<select id="post_tags" name="post_tags[]" class="form-control" multiple="multiple"></select>
												</div>
											</div>

											<div class="col-12">
												<h3 class="kanews-custom-page-subtitle mt-1"><?php esc_html_e( 'İlan Hakkında Daha Fazla Bilgi', 'kanews' ); ?></h3>
												<div class="repeater">
													<div data-repeater-list="ilan_list">
														<div class="row" data-repeater-item>

																<div class="col">
																<div class="kanews-form">
																<input required type="text" name="title" class="form-control" value="">
																<label><?php _e('Başlık', 'kanews'); ?></label>

																</div>
																</div>
											
											
																<div class="col">
																<div class="kanews-form">
																	<input required type="text" name="text" class="form-control" value="" ></input>
																	<button data-repeater-delete type="button" class="kanews-btn kanews-btn-sm pull-right"><?php esc_html_e('Kaldır', 'kanews'); ?></button>
																	<label><?php _e('Detay', 'kanews'); ?></label>

																</div>
																</div>
													
															
														</div>
													</div>
													<input data-repeater-create type="button" class="kanews-btn kanews-btn-block kanews-btn-sm bg-info" value="<?php esc_html_e('Yeni Ekle', 'kanews'); ?>"/>
												</div>
												<script data-no-optimize="1" src="//ajax.googleapis.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>

												<script data-no-optimize="1" src="https://cdnjs.cloudflare.com/ajax/libs/jquery.repeater/1.2.1/jquery.repeater.min.js"></script>
												<script>
													$('.repeater').repeater({
														show: function () {
															$(this).slideDown();
														},
														// initEmpty: true,
														isFirstItemUndeletable: true
													});

												</script>
											</div>

										</div>
										<h3 class="kanews-custom-page-subtitle mt-1 mb-0"><?php esc_html_e( 'İlan Hakkında', 'kanews' ); ?></h3>
										<?php 
											$content = '';
											$editor_id = 'editor';
											$settings =   array(
													'wpautop' => true,
													'media_buttons' => false, 
													'textarea_name' => $editor_id, 
													'textarea_rows' => get_option('default_post_edit_rows', 10),
													'tabindex' => '', 
													'editor_css' => '', 
													'editor_class' => '', 
													'teeny' => false, 
													'dfw' => false,
													'tinymce' => true, 
													'quicktags' => true
											);
											wp_editor( $content, $editor_id, $settings );
										?>

											<script type="text/javascript" data-no-optimize="1" src="https://maps.googleapis.com/maps/api/js?key=AIzaSyC5Jrp9PtHe0WapppUzxbIpMDWMAcV3qE4"></script>
  										<script data-no-optimize="1"src="https://unpkg.com/location-picker/dist/location-picker.min.js"></script>
											<div class="col-12">
											<h3 class="kanews-custom-page-subtitle mt-1"><?php esc_html_e( 'Harita Konumu', 'kanews' ); ?></h3>
												<div id="map" class="mb-1" style=" width: 100%;
														height: 480px;"></div>
											</div>

											<script>
											// Get element references
											var confirmBtn = document.getElementById('confirmPosition');
											var onClickPositionView = document.getElementById('onClickPositionView');
											var onIdlePositionView = document.getElementById('ilan_map');
											var map = document.getElementById('map');

											// Initialize LocationPicker plugin
											var lp = new locationPicker(map, {
											setCurrentPosition: true, // You can omit this, defaults to true
											lat: 41.02204230308628,
											lng: 28.995666903485418
											}, {
											zoom: 13 // You can set any google map options here, zoom defaults to 15
											});


											// Listen to map idle event, listening to idle event more accurate than listening to ondrag event
											google.maps.event.addListener(lp.map, 'idle', function (event) {
											// Get current location and show it in HTML
											var location = lp.getMarkerPosition();
											onIdlePositionView.value = 'https://maps.google.com/maps?q='+location.lat +','+location.lng+'&hl=es&z=14&amp;output=embed'
											});
											</script>


										<div class="row">
											<div class="col-12">
												<h3 class="kanews-custom-page-subtitle mt-1"><?php esc_html_e( 'İlanın Öne Çıkarılan Görseli', 'kanews' ); ?></h3>
												<input type="file" name="featured" id="featured" accept="image/*" />
											</div>

											<?php for ($x = 1; $x <= 5; $x++) { ?>
												<div class="col-12">
													<h3 class="kanews-custom-page-subtitle mt-1"><?php esc_html_e( 'İlan Görseli', 'kanews' ); ?> #<?php echo $x; ?></h3>
													<input type="file" name="ilan_thumb<?php echo $x; ?>" id="ilan_thumb<?php echo $x; ?>" accept="image/*" />
												</div>
											<?php } ?>
		
											<?php echo kanews_get_recaptcha('registration_form'); ?>
											<div class="col-12 mt-2">
												<button class="kanews-btn kanews-btn-block" name="ilan-submit" type="submit"><?php esc_html_e( 'Yayınla', 'kanews' ); ?> <div id="kanews-loader"></div></button>
											</div>
									
										</div>
								</form>
								<?php else: echo esc_html_e( 'Bu sayfa geçici olarak devre dışı bırakılmıştır.', 'kanews' ); endif; break;
								case 'firma-ekle': if( kanews_get_option('firma-rehberi') ): ?>
									<h2 class="kanews-custom-page-title"><?php esc_html_e( 'Firma Ekle', 'kanews' ); ?></h2>
									<p class="mb-3"><?php echo kanews_translate('kanews-add-post-text', esc_html_e( 'Firmanız editörümüz tarafından incelendikten sonra uygun görüldüğü takdirde yayınlanacaktır.', 'kanews' )); ?></p>
									<form action="<?php the_permalink(); ?>?islem=firma-ekle" method="post" enctype="multipart/form-data">
										<?php echo $error; ?>
										<div class="row">
											<div class="col-12">
												<div class="kanews-form">      
													<input type="text" id="comp_name" name="comp_name" value="<?php echo isset($_POST['comp_name']) ? esc_attr($_POST['comp_name'])  : ''; ?>" required autofocus>
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('Firma Adı', 'kanews'); ?> <span class="color-danger">*</span></label>
												</div>
											</div>
											<div class="col-12 col-md-6">
												<div class="kanews-form">      
													<input type="text" id="comp_auth" name="comp_auth"value="<?php echo isset($_POST['comp_auth'])  ? esc_attr($_POST['comp_auth'])  : ''; ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('Yetkili Adı', 'kanews'); ?></label>
												</div>
											</div>
											<div class="col-12 col-md-6">
												<div class="kanews-form">      
													<input type="text" id="comp_mail" name="comp_mail" value="<?php echo isset($_POST['comp_mail'])  ? esc_attr($_POST['comp_mail'])  : ''; ?>" required>
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('E-Posta', 'kanews'); ?> <span class="color-danger">*</span></label>
												</div>
											</div>
											<div class="col-12 col-md-6">
												<div class="kanews-form">      
													<input type="text" id="comp_city" name="comp_city" value="<?php echo isset($_POST['comp_city'])  ? esc_attr($_POST['comp_city'])  : ''; ?>" required>
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('Şehir', 'kanews'); ?> <span class="color-danger">*</span></label>
												</div>
											</div>
											<div class="col-12 col-md-6">
												<div class="kanews-form">      
													<input type="text" id="comp_site" name="comp_site" value="<?php echo isset($_POST['comp_site'])  ? esc_attr($_POST['comp_site'])  : ''; ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('Web Sitesi', 'kanews'); ?></label>
												</div>
											</div>
		
											<div class="col-12">
												<div class="kanews-form">      
													<input type="text" id="comp_adress" name="comp_adress" value="<?php isset($_POST['comp_adress']) ? esc_attr($_POST['comp_adress'])  : ''; ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('Tam Adres', 'kanews'); ?></label>
												</div>
											</div>
											<div class="col-12 col-md-6">
												<div class="kanews-form">      
													<input type="text" id="comp_tel" name="comp_tel" value="<?php echo isset($_POST['comp_tel']) ? esc_attr($_POST['comp_tel'])  : ''; ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('Telefon Numarası', 'kanews'); ?></label>
												</div>
											</div>
											<div class="col-12 col-md-6">
												<div class="kanews-form">      
													<input type="text" id="comp_fax" name="comp_fax" value="<?php echo isset($_POST['comp_fax']) ? esc_attr($_POST['comp_fax'])  : ''; ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('Faks Numarası', 'kanews'); ?></label>
												</div>
											</div>

											<div class="col-12" hidden>
												<div class="kanews-form">      
													<input value="" type="text" id="comp_map" name="comp_map" value="<?php isset($_POST['comp_map']) ? $_POST['comp_map']  : ''; ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('Harita Embed Linki', 'kanews'); ?></label>
												</div>
											</div>

											<div class="col-12 mb-1">
												<div class="kanews-form">
													<?php wp_dropdown_categories( 'show_option_none='.__('Kategori Seç', 'kanews').'&tab_index=4&taxonomy=kategori&multiple=1&hide_empty=0' ); ?>
												</div>
											</div>


											<script type="text/javascript" data-no-optimize="1" src="https://maps.googleapis.com/maps/api/js?key=AIzaSyC5Jrp9PtHe0WapppUzxbIpMDWMAcV3qE4"></script>
  										<script data-no-optimize="1"src="https://unpkg.com/location-picker/dist/location-picker.min.js"></script>
											<div class="col-12">
											<h3 class="kanews-custom-page-subtitle mt-1"><?php esc_html_e( 'Harita Konumu', 'kanews' ); ?></h3>
												<div id="map" class="mb-1" style=" width: 100%;
														height: 480px;"></div>
											</div>

											<script>
											// Get element references
											var confirmBtn = document.getElementById('confirmPosition');
											var onClickPositionView = document.getElementById('onClickPositionView');
											var onIdlePositionView = document.getElementById('comp_map');
											var map = document.getElementById('map');

											// Initialize LocationPicker plugin
											var lp = new locationPicker(map, {
											setCurrentPosition: true, // You can omit this, defaults to true
											lat: 41.02204230308628,
											lng: 28.995666903485418
											}, {
											zoom: 13 // You can set any google map options here, zoom defaults to 15
											});


											// Listen to map idle event, listening to idle event more accurate than listening to ondrag event
											google.maps.event.addListener(lp.map, 'idle', function (event) {
											// Get current location and show it in HTML
											var location = lp.getMarkerPosition();
											onIdlePositionView.value = 'https://maps.google.com/maps?q='+location.lat +','+location.lng+'&hl=es&z=14&amp;output=embed'
											});
											</script>

										</div>
										<h3 class="kanews-custom-page-subtitle mt-1 mb-0"><?php esc_html_e( 'Firma Hakkında', 'kanews' ); ?></h3>
										<?php 
											$content = '';
											$editor_id = 'editor';
											$settings =   array(
													'wpautop' => true,
													'media_buttons' => false, 
													'textarea_name' => $editor_id, 
													'textarea_rows' => get_option('default_post_edit_rows', 10),
													'tabindex' => '', 
													'editor_css' => '', 
													'editor_class' => '', 
													'teeny' => false, 
													'dfw' => false,
													'tinymce' => true, 
													'quicktags' => true
											);
											wp_editor( $content, $editor_id, $settings );
										?>
										<div class="row">
											<div class="col-12">
												<h3 class="kanews-custom-page-subtitle mt-1"><?php esc_html_e( 'Firma Logosu', 'kanews' ); ?></h3>
												<input required type="file" name="featured" id="featured" accept="image/*" />
											</div>
											<?php echo kanews_get_recaptcha('registration_form'); ?>
											<div class="col-12 mt-2">
												<button class="kanews-btn kanews-btn-block" name="firma-submit" type="submit"><?php esc_html_e( 'Yayınla', 'kanews' ); ?> <div id="kanews-loader"></div></button>
											</div>
									
										</div>
									</form>
									<?php else: echo esc_html_e( 'Bu sayfa geçici olarak devre dışı bırakılmıştır.', 'kanews' ); endif; break;
								case 'firma-duzenle': 
									if(kanews_get_option('firma-edit') && kanews_get_option('firma-rehberi') && isset($_GET['id'])): $post_id = $_GET['id']; $content = get_post( $post_id, OBJECT, 'edit' ); ?>
									<h2 class="kanews-custom-page-title"><?php esc_html_e( 'Firma Düzenle', 'kanews' ); ?></h2>
									<?php $user_meta = get_userdata($current_user->ID);
									if($user_meta->roles[0] == 'subscriber'): 
										if('pending' == get_post_status ( $_GET['id'] )) : ?>
											<div class="kanews-response"><?php esc_html_e('Yayınlanması için yönetici onayı bekleniyor.', 'kanews'); ?></div>
										<?php else: ?>
											<div class="kanews-response"><?php esc_html_e('Kaydederseniz yayından kaldırılacak ve tekrar yayınlanmak için yönetici onayı beklenecektir.', 'kanews'); ?></div>
										<?php endif; ?>
									<?php endif; ?>
									<?php if( ! isset($_GET['id']) || get_post_type($_GET['id']) != 'firma' || ( 'publish' != get_post_status ( $_GET['id'] ) && 'pending' != get_post_status ( $_GET['id'] ) ) || ($content->post_author != $current_user->ID ) ) : ?>
										<p><?php esc_html_e( 'Bu içeriği düzenleme yetkiniz bulunmamaktadır.', 'kanews' ); ?></p>
									<?php else:  ?>
										<form action="<?php the_permalink(); ?>?islem=firma-duzenle&id=<?php echo $post_id; ?>" method="post" enctype="multipart/form-data">
											<?php echo $error; ?>
											<div class="row">
												<div class="col-12">
													<div class="kanews-form">      
														<input type="text" id="comp_name" name="comp_name" value="<?php echo esc_attr(get_post_meta($post_id, 'comp_name', true)); ?>" required autofocus>
														<span class="highlight"></span>
														<span class="bar"></span>
														<label><?php _e('Firma Adı', 'kanews'); ?> <span class="color-danger">*</span></label>
													</div>
												</div>
												<div class="col-12 col-md-6">
													<div class="kanews-form">      
														<input type="text" id="comp_auth" name="comp_auth"value="<?php echo esc_attr(get_post_meta($post_id, 'comp_auth', true)); ?>">
														<span class="highlight"></span>
														<span class="bar"></span>
														<label><?php _e('Yetkili Adı', 'kanews'); ?></label>
													</div>
												</div>
												<div class="col-12 col-md-6">
													<div class="kanews-form">      
														<input type="text" id="comp_mail" name="comp_mail" value="<?php echo get_post_meta($post_id, 'comp_mail', true); ?>" required>
														<span class="highlight"></span>
														<span class="bar"></span>
														<label><?php _e('E-Posta', 'kanews'); ?> <span class="color-danger">*</span></label>
													</div>
												</div>
												<div class="col-12 col-md-6">
													<div class="kanews-form">      
														<input type="text" id="comp_city" name="comp_city" value="<?php echo esc_attr(get_post_meta($post_id, 'comp_city', true)); ?>" required>
														<span class="highlight"></span>
														<span class="bar"></span>
														<label><?php _e('Şehir', 'kanews'); ?> <span class="color-danger">*</span></label>
													</div>
												</div>
												<div class="col-12 col-md-6">
													<div class="kanews-form">      
														<input type="text" id="comp_site" name="comp_site" value="<?php echo esc_attr(get_post_meta($post_id, 'comp_site', true)); ?>">
														<span class="highlight"></span>
														<span class="bar"></span>
														<label><?php _e('Web Sitesi', 'kanews'); ?></label>
													</div>
												</div>
												<div class="col-12">
													<div class="kanews-form">      
														<input value="<?php echo esc_url(get_post_meta($post_id, 'comp_map', true)); ?>" class="used" placeholder="https://www.google.com/maps/embed?pb=!1m14!1m8!1m3!1d1530.4036739738247!2d32.709943736128224!3d39.900945406411594!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x14d3387b0317bdb5%3A0x49a13bb3b5b5ec48!2zQUTEsFZBIFVNxLBUS09Z!5e0!3m2!1str!2str!4v1615728946790!5m2!1str!2str" type="text" id="comp_map" name="comp_map" value="<?php isset($_POST['comp_map']) ? $_POST['comp_map']  : ''; ?>">
														<span class="highlight"></span>
														<span class="bar"></span>
														<label><?php _e('Harita Embed Linki', 'kanews'); ?></label>
													</div>
												</div>
												<div class="col-12">
													<div class="kanews-form">      
														<input type="text" id="comp_adress" name="comp_adress" value="<?php echo esc_attr(get_post_meta($post_id, 'comp_adress', true)); ?>">
														<span class="highlight"></span>
														<span class="bar"></span>
														<label><?php _e('Tam Adres', 'kanews'); ?></label>
													</div>
												</div>

												<div class="col-12 col-md-6">
													<div class="kanews-form">      
														<input type="text" id="comp_tel" name="comp_tel" value="<?php echo esc_attr(get_post_meta($post_id, 'comp_tel', true)); ?>">
														<span class="highlight"></span>
														<span class="bar"></span>
														<label><?php _e('Telefon Numarası', 'kanews'); ?></label>
													</div>
												</div>
												<div class="col-12 col-md-6">
													<div class="kanews-form">      
														<input type="text" id="comp_fax" name="comp_fax" value="<?php echo esc_attr(get_post_meta($post_id, 'comp_fax', true)); ?>">
														<span class="highlight"></span>
														<span class="bar"></span>
														<label><?php _e('Faks Numarası', 'kanews'); ?></label>
													</div>
												</div>


											</div>
											<h3 class="kanews-custom-page-subtitle mt-1 mb-0"><?php esc_html_e( 'Firma Hakkında', 'kanews' ); ?></h3>
											<?php 
												$content = $content->post_content;
												$editor_id = 'editor';
												$settings =   array(
														'wpautop' => true,
														'media_buttons' => false, 
														'textarea_name' => $editor_id, 
														'textarea_rows' => get_option('default_post_edit_rows', 10),
														'tabindex' => '', 
														'editor_css' => '', 
														'editor_class' => '', 
														'teeny' => false, 
														'dfw' => false,
														'tinymce' => true, 
														'quicktags' => true
												);
												wp_editor( $content, $editor_id, $settings );
											?>
											<div class="row">
												<div class="col-12">
													<h3 class="kanews-custom-page-subtitle mt-1"><?php esc_html_e( 'Firma Logosu', 'kanews' ); ?></h3>
													<input type="file" name="featured" id="featured" accept="image/*" />
												</div>

												<div class="col-12 mt-2">
												<button class="kanews-btn kanews-btn-block" name="firma-edit" type="submit"><?php esc_html_e( 'Düzenle', 'kanews' ); ?> <div id="kanews-loader"></div></button>
												<a class="kanews-btn kanews-btn-sm kanews-btn-block bg-dark" onclick="return confirm('Gerçekten Silmek istiyor musun?')" href="<?php the_permalink(); ?>?islem=firma-duzenle&id=<?php echo $post_id; ?>&r=1"><?php esc_html_e('Yayından Kaldır', 'kanews'); ?></a>
											</div>
											
												<?php echo kanews_get_recaptcha('registration_form'); ?>
									
										
											</div>
										</form>
									<?php endif; ?>
									<?php else: echo esc_html_e( 'Bu sayfa geçici olarak devre dışı bırakılmıştır.', 'kanews' ); endif; break;
								case 'icerik-duzenle': if(kanews_get_option('post-edit') && isset($_GET['id'])): $post_id = $_GET['id']; $content = get_post( $post_id, OBJECT, 'edit' ); ?>
									<link href="//cdn.quilljs.com/1.3.6/quill.snow.css" rel="stylesheet">
									<h2 class="kanews-custom-page-title mb-2"><?php esc_html_e( 'İçerik Düzenle', 'kanews' ); ?></h2>
									<?php $user_meta = get_userdata($current_user->ID);
									if($user_meta->roles[0] == 'subscriber'): 
										if('pending' == get_post_status ( $_GET['id'] )) : ?>
											<div class="kanews-response"><?php esc_html_e('Yayınlanması için yönetici onayı bekleniyor.', 'kanews'); ?></div>
										<?php else: ?>
											<div class="kanews-response"><?php esc_html_e('Kaydederseniz yayından kaldırılacak ve tekrar yayınlanmak için yönetici onayı beklenecektir.', 'kanews'); ?></div>
										<?php endif; ?>
									<?php endif; ?>
									<?php if( ! isset($_GET['id']) || ( get_post_type($_GET['id']) != 'post' && get_post_type($_GET['id']) != 'columist' ) || ( 'publish' != get_post_status ( $_GET['id'] ) && 'pending' != get_post_status ( $_GET['id'] ) ) || ($content->post_author != $current_user->ID ) ) : ?>
										<p><?php esc_html_e( 'Bu içeriği düzenleme yetkiniz bulunmamaktadır.', 'kanews' ); ?></p>
									<?php else:  ?>
				
									<form action="<?php the_permalink(); ?>?islem=icerik-duzenle&id=<?php echo $post_id; ?>" method="post" id="post-submit" enctype="multipart/form-data">
										<?php echo $error; ?>
										<div class="row">

											<div class="col-12">
												<div class="kanews-form">      
													<input type="text" id="post_title" name="post_title" value="<?php echo esc_attr($content->post_title); ?>" required autofocus>

													<label><?php _e('Başlık', 'kanews'); ?></label>
												</div>
											</div>

											<div class="col-12">
												<div class="kanews-form">      
													<textarea type="text" id="post_excerpt" name="post_excerpt"><?php echo wp_strip_all_tags(get_post_field('post_excerpt', $post_id)); ?></textarea>
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('Açıklama', 'kanews'); ?></label>
												</div>
											</div>

											<didivlass= class="col-12 mb-1">
												<div class="kanews-form">
													<?php wp_dropdown_categories( 'show_option_none='.__('Kategori Seç', 'kanews').'&selected='.Kanews\Helper_Wp::primary_category_id($post_id).'&tab_index=4&taxonomy=category&multiple=1' ); ?>
												</div>
											</didivlass=>

						
											
										</div>

										<div id="editor" class="kanews-article-content"><?php echo wpautop($content->post_content); ?></div>
										<textarea name="content" style="display:none" id="content"><?php echo $content->post_content; ?></textarea>

										<div id="image-preview" class="mt-1" style="position:relative;">
											<div class="kanews-modal-close-btn" ><i class="icon-close"></i></div>
											<?php if(has_post_thumbnail($post_id)): ?>
												<img style="width: 100%; height: auto" src="<?php echo wp_get_attachment_image_src( get_post_thumbnail_id($content), 'full' )[0]; ?>" class="preview-image">
											<?php endif; ?>
										</div>

										<div class="row">
											<div class="col-12 mt-1">
												
												<input type="file" name="featured" id="featured" accept="image/*" class="kanews-input-file"><label for="featured" class="kanews-input-file-path"><?php esc_html_e( 'Öne Çıkarılan Görsel Ekle', 'kanews' ); ?></label></input>
											</div>
											<?php echo kanews_get_recaptcha('registration_form'); ?>
											<div class="col-12">
												<button class="kanews-btn kanews-btn-block" name="post-edit" type="submit"><?php esc_html_e( 'Değişiklikleri Kaydet', 'kanews' ); ?> <div id="kanews-loader"></div></button>
											</div>
										</div>
									</form>
									<script data-no-optimize="1" src="//cdn.quilljs.com/1.3.6/quill.min.js"></script>
									<script>
									// editor
									var toolbarOptions = [
										[{ 'header': [1, 2, 3, 4, 5, 6, false] }],
										['bold', 'italic', 'underline'],
										[{ 'list': 'ordered'}, { 'list': 'bullet' }],
										['link'],
										['image'],
										['clean']                                         // remove formatting button
									];
									var quill = new Quill('#editor', {
									modules: {
										toolbar: toolbarOptions
									},
									theme: 'snow'
									});
										const file = document.getElementById('featured'),
													image = document.querySelector('.preview-image')

										file.addEventListener('change', function() {
											[... this.files].map(file => {
												if (file.name.toLowerCase().match(/\.(jpe?g|png|webp|gif)$/)) {
													const reader = new FileReader()
													reader.addEventListener('load', function () {
														image.src = this.result
													})
													reader.readAsDataURL(file)
													document.getElementById("image-preview").style.display = "block";
												}
											})
										})


										document.querySelector('#image-preview .kanews-modal-close-btn').onclick = function(){
											file.value=''
											image.src=''
											document.getElementById("image-preview").style.display = "none";
										}
									</script>
									
									<?php endif; ?>
									
					

									<?php else: echo esc_html_e( 'Bu sayfa geçici olarak devre dışı bırakılmıştır.', 'kanews' ); endif; break;
								case 'ilan-duzenle': 
								if(kanews_get_option('ilan-edit') && kanews_get_option('ilan-sistemi') && isset($_GET['id'])): $post_id = $_GET['id']; $content = get_post( $post_id, OBJECT, 'edit' ); ?>
									<h2 class="kanews-custom-page-title mb-2"><?php esc_html_e( 'İlan Düzenle', 'kanews' ); ?></h2>
									<?php $user_meta = get_userdata($current_user->ID);
									if($user_meta->roles[0] == 'subscriber'): 
										if('pending' == get_post_status ( $_GET['id'] )) : ?>
											<div class="kanews-response"><?php esc_html_e('Yayınlanması için yönetici onayı bekleniyor.', 'kanews'); ?></div>
										<?php else: ?>
											<div class="kanews-response"><?php esc_html_e('Kaydederseniz yayından kaldırılacak ve tekrar yayınlanmak için yönetici onayı beklenecektir.', 'kanews'); ?></div>
										<?php endif; ?>
									<?php endif; ?>
									<?php if( ! isset($_GET['id']) || get_post_type($_GET['id']) != 'ilan' || ( 'publish' != get_post_status ( $_GET['id'] ) && 'pending' != get_post_status ( $_GET['id'] ) ) || ($content->post_author != $current_user->ID ) ) : ?>
										<p><?php esc_html_e( 'Bu içeriği düzenleme yetkiniz bulunmamaktadır.', 'kanews' ); ?></p>
									<?php else:  ?>
										<form action="<?php the_permalink(); ?>?islem=ilan-duzenle&id=<?php echo $post_id; ?>" method="post" enctype="multipart/form-data">
										<?php echo $error; ?>
		
										
  									<div class="row">
											<div class="col-12">
												<div class="kanews-form">      
													<input type="text" id="ilan_name" name="ilan_name" value="<?php echo esc_attr($content->post_title); ?>" required autofocus>
						
													<label><?php _e('İlan Adı', 'kanews'); ?> <span class="color-danger">*</span></label>
												</div>
											</div>
											<?php 
												// Taxonomy'den il ve ilçe değerlerini al
												$ilan_il_terms = wp_get_post_terms($post_id, 'ilan-il', array('fields' => 'names'));
												$ilan_ilce_terms = wp_get_post_terms($post_id, 'ilan-ilce', array('fields' => 'names'));
												
												$ilan_il = !empty($ilan_il_terms) && !is_wp_error($ilan_il_terms) ? $ilan_il_terms[0] : '';
												$ilan_ilce = !empty($ilan_ilce_terms) && !is_wp_error($ilan_ilce_terms) ? $ilan_ilce_terms[0] : '';
												
												$ilan_il_plaka = $ilan_il ? kanews_get_plaka($ilan_il) : '';
												// Eğer plaka bulunamadıysa (Almanya, KKTC gibi), şehir ismini kullan
												if(empty($ilan_il_plaka) || $ilan_il_plaka == 0) {
													$ilan_il_plaka = $ilan_il;
												}
											?>
											<div class="col-12 col-md-6">
												<div class="kanews-form2 bgw position-relative">
													<input type="text" 
														   id="dashboard-ilan-edit-il-search" 
														   class="mb-0" 
														   placeholder="<?php esc_html_e( 'Ara', 'kanews' ); ?>" 
														   autocomplete="off"
														   value="<?php echo $ilan_il ? esc_attr($ilan_il) : ''; ?>" />
													<button type="button" class="ilan-clear-btn" id="dashboard-ilan-edit-il-clear" style="display: <?php echo $ilan_il ? 'block' : 'none'; ?>;">
														<i class="icon-close"></i>
													</button>
													<select id="dashboard-ilan-edit-il" style="display: none;">
														<option value=""><?php esc_html_e( 'İl Seç', 'kanews' ); ?></option>
													</select>
													<input type="hidden" name="il" id="dashboard-ilan-edit-il-hidden" value="<?php echo $ilan_il_plaka ? esc_attr($ilan_il_plaka) : ''; ?>" />
													<div id="dashboard-ilan-edit-il-dropdown" class="ilan-dropdown" style="display: none;"></div>
												</div>
											</div>
											<div class="col-12 col-md-6">
												<div class="kanews-form2 bgw position-relative">
													<input type="text" 
														   id="dashboard-ilan-edit-ilce-search" 
														   class="mb-0" 
														   placeholder="<?php esc_html_e( 'Ara', 'kanews' ); ?>" 
														   autocomplete="off"
														   value="<?php echo $ilan_ilce ? esc_attr($ilan_ilce) : ''; ?>"
														   <?php echo !$ilan_il ? 'disabled' : ''; ?> />
													<button type="button" class="ilan-clear-btn" id="dashboard-ilan-edit-ilce-clear" style="display: <?php echo $ilan_ilce ? 'block' : 'none'; ?>;">
														<i class="icon-close"></i>
													</button>
													<select id="dashboard-ilan-edit-ilce" style="display: none;">
														<option value=""><?php esc_html_e( 'İlçe Seç', 'kanews' ); ?></option>
													</select>
													<input type="hidden" name="ilce" id="dashboard-ilan-edit-ilce-hidden" value="<?php echo $ilan_ilce ? esc_attr($ilan_ilce) : ''; ?>" />
													<div id="dashboard-ilan-edit-ilce-dropdown" class="ilan-dropdown" style="display: none;"></div>
												</div>
											</div>
											
											
											<div class="col-12">
												<div class="kanews-form">      
													<input type="text" id="ilan_adress" name="ilan_adress" value="<?php echo esc_attr(get_post_meta($post_id, 'ilan_adress',true)); ?>">

													<label><?php _e('Tam Adres', 'kanews'); ?></label>
												</div>
											</div>
											<div class="col-12">
												<div class="kanews-form">      
													<input placeholder="https://www.google.com/maps/embed?pb=!1m14!1m8!1m3!1d1530.4036739738247!2d32.709943736128224!3d39.900945406411594!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x14d3387b0317bdb5%3A0x49a13bb3b5b5ec48!2zQUTEsFZBIFVNxLBUS09Z!5e0!3m2!1str!2str!4v1615728946790!5m2!1str!2str" type="text" id="ilan_map" name="ilan_map" value="<?php echo esc_url(get_post_meta($post_id, 'ilan_map',true)); ?>">
													<label><?php _e('Harita Embed Linki', 'kanews'); ?></label>
												</div>
											</div>
											<div class="col-12">
												<div class="kanews-form">      
													<input type="text" id="ilan_price" name="ilan_price" value="<?php echo esc_attr(get_post_meta($post_id, 'ilan_price',true)); ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('İlan Fiyatı', 'kanews'); ?></label>
												</div>
											</div>
											<div class="col-12 col-md-6">
												<div class="kanews-form">      
													<input type="text" id="ilan_tel" name="ilan_tel" value="<?php echo esc_attr(get_post_meta($post_id, 'ilan_tel',true)); ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('Telefon Numarası', 'kanews'); ?></label>
												</div>
											</div>
											<div class="col-12 col-md-6">
												<div class="kanews-form">      
													<input type="text" id="ilan_tel2" name="ilan_tel2" value="<?php echo esc_attr(get_post_meta($post_id, 'ilan_tel2',true)); ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('Telefon Numarası', 'kanews'); ?> 2</label>
												</div>
											</div>
                                            <div class="col-12 col-md-6">
                                                <div class="kanews-form">      
                                                    <input type="text" id="ilan_whatsapp" name="ilan_whatsapp" value="<?php echo esc_attr(get_post_meta($post_id, 'ilan_whatsapp',true)); ?>">
                                                    <span class="highlight"></span>
                                                    <span class="bar"></span>
                                                    <label><?php _e('Whatsapp Numarası', 'kanews'); ?></label>
                                                    <small class="text-muted" style="display:block;margin-top:4px;"><?php esc_html_e('wa.me’den sonrasını yazın, örn: 905551112233', 'kanews'); ?></small>
                                                </div>
                                            </div>
											<div class="col-12 col-md-6">
												<div class="kanews-form">      
													<input type="email" id="ilan_mail" name="ilan_mail" value="<?php echo esc_attr(get_post_meta($post_id, 'ilan_mail',true)); ?>">
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('E-Posta', 'kanews'); ?></label>
												</div>
											</div>

										</div>
										<h3 class="kanews-custom-page-subtitle mt-1 mb-0"><?php esc_html_e( 'İlan Hakkında', 'kanews' ); ?></h3>
										<?php 
											$content = $content->post_content;
											$editor_id = 'editor';
											$settings =   array(
													'wpautop' => true,
													'media_buttons' => false, 
													'textarea_name' => $editor_id, 
													'textarea_rows' => get_option('default_post_edit_rows', 10),
													'tabindex' => '', 
													'editor_css' => '', 
													'editor_class' => '', 
													'teeny' => false, 
													'dfw' => false,
													'tinymce' => true, 
													'quicktags' => true
											);
											wp_editor( $content, $editor_id, $settings );
										?>
										<div class="row">

										<div class="col-12">
												<h3 class="kanews-custom-page-subtitle mt-1"><?php esc_html_e( 'İlan Hakkında Daha Fazla Bilgi', 'kanews' ); ?></h3>
												<div class="repeater">
													<div data-repeater-list="ilan_list">
														<?php 
															$ilan_list = get_post_meta($post_id, 'ilan_list', true);
															if( is_array($ilan_list) ) {
														foreach(get_post_meta($post_id, 'ilan_list', true) as $item): ?>
														<div class="row" data-repeater-item>

																<div class="col">
																<div class="kanews-form">
																<input required type="text" name="title" class="form-control" value="<?php echo isset($item['title']) ? $item['title'] : ''; ?>">
																<label><?php _e('Başlık', 'kanews'); ?></label>

																</div>
																</div>
											
											
																<div class="col">
																<div class="kanews-form">
																	<input required type="text" name="text" class="form-control" value="<?php echo isset($item['text']) ? $item['text'] : ''; ?>" ></input>
																	<button data-repeater-delete type="button" class="kanews-btn kanews-btn-sm pull-right"><?php esc_html_e('Kaldır', 'kanews'); ?></button>
																	<label><?php _e('Detay', 'kanews'); ?></label>

																</div>
																</div>
													
															
														</div>
														<?php endforeach; 
															} else {
																?>
																<div class="row" data-repeater-item>

<div class="col">
<div class="kanews-form">
<input required type="text" name="title" class="form-control" value="">
<label><?php _e('Başlık', 'kanews'); ?></label>

</div>
</div>


<div class="col">
<div class="kanews-form">
	<input required type="text" name="text" class="form-control" value="" ></input>
	<button data-repeater-delete type="button" class="kanews-btn kanews-btn-sm pull-right"><?php esc_html_e('Kaldır', 'kanews'); ?></button>
	<label><?php _e('Detay', 'kanews'); ?></label>

</div>
</div>


</div>
																<?php 
															}
														?>
													</div>
													<input data-repeater-create type="button" class="kanews-btn kanews-btn-block kanews-btn-sm bg-info" value="<?php esc_html_e('Yeni Ekle', 'kanews'); ?>"/>
												</div>
												<script data-no-optimize="1" src="//ajax.googleapis.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>

												<script data-no-optimize="1" src="https://cdnjs.cloudflare.com/ajax/libs/jquery.repeater/1.2.1/jquery.repeater.min.js"></script>
												<script>
													$('.repeater').repeater({
														show: function () {
															$(this).slideDown();
														},
														// initEmpty: true,
														isFirstItemUndeletable: true
													});

												</script>
											</div>

											<div class="col-12">
												<h3 class="kanews-custom-page-subtitle mt-1"><?php esc_html_e( 'İlanın Öne Çıkarılan Görseli', 'kanews' ); ?></h3>
												<input type="file" name="featured" id="featured" accept="image/*" />
											</div>

											<?php for ($x = 1; $x <= 5; $x++) { ?>
												<div class="col-12">
													<h3 class="kanews-custom-page-subtitle mt-1"><?php esc_html_e( 'İlan Görseli', 'kanews' ); ?> #<?php echo $x; ?></h3>
													<input type="file" name="ilan_thumb<?php echo $x; ?>" id="ilan_thumb<?php echo $x; ?>" accept="image/*" />
												</div>
											<?php } ?>
		
											<?php echo kanews_get_recaptcha('registration_form'); ?>
											<div class="col-12 mt-2">
												<button class="kanews-btn kanews-btn-block" name="ilan-edit" type="submit"><?php esc_html_e( 'Düzenle', 'kanews' ); ?> <div id="kanews-loader"></div></button>
												<a class="kanews-btn kanews-btn-sm kanews-btn-block bg-dark" onclick="return confirm('Gerçekten Silmek istiyor musun?')" href="<?php the_permalink(); ?>?islem=ilan-duzenle&id=<?php echo $post_id; ?>&r=1"><?php esc_html_e('Yayından Kaldır', 'kanews'); ?></a>
											</div>
									
										</div>
								</form>
								<?php endif; ?>
				
								<?php else: echo esc_html_e( 'Bu sayfa geçici olarak devre dışı bırakılmıştır.', 'kanews' ); endif; break;
								case 'kose-yazisi-ekle':
								case 'icerik-ekle':
								if(kanews_get_option('post-submit') == true): ?>

									<link href="//cdn.quilljs.com/1.3.6/quill.snow.css" rel="stylesheet">
									<h2 class="kanews-custom-page-title"><?php echo $action2 == 'kose-yazisi-ekle' ? esc_html__( 'Köşe Yazısı Ekle', 'kanews' ) : esc_html__( 'İçerik Ekle', 'kanews' ); ?></h2>
									<p class="mb-3"><?php echo kanews_translate('kanews-add-post-text', esc_html_e( 'Paylaştığınız içerik editörümüz tarafından incelendikten sonra uygun görüldüğü takdirde yayınlanacaktır.', 'kanews' )); ?></p>
									<?php $disabled = false; if($action2 == 'kose-yazisi-ekle' && kanews_get_option('columnist-submit-only')) {
										$user_meta = get_userdata($user_id);
										if($user_meta->roles[0] != 'columist')
										$disabled = true;
									} ?>
									<?php if ( ! $disabled ) : ?>
									<form action="<?php the_permalink(); ?>?islem=<?php echo $action2; ?>" method="post" id="post-submit" enctype="multipart/form-data">
									<?php if(!empty($error)): ?>
										<?php echo $error; ?>
										<?php endif; ?>
										<div class="row">

											<div class="col-12">
												<div class="kanews-form">      
													<input type="text" id="post_title" name="post_title" value="" required autofocus>
													<span class="highlight"></span>
													<span class="bar"></span>
													<label><?php _e('Başlık', 'kanews'); ?></label>
												</div>
											</div>

											<?php if($action2 != 'kose-yazisi-ekle'): ?>
												<div class="col-12">
													<div class="kanews-form">      
														<textarea type="text" id="post_excerpt" name="post_excerpt"></textarea>
														<span class="highlight"></span>
														<span class="bar"></span>
														<label><?php _e('Açıklama', 'kanews'); ?></label>
													</div>
												</div>

												

												<div class="col-12 mb-1">
													<div class="kanews-form">
														<?php wp_dropdown_categories( 'show_option_none='.__('Kategori Seç', 'kanews').'&tab_index=4&taxonomy=category&multiple=1&hide_empty=0' ); ?>
													</div>
												</div>

												<div class="col-12">
													<div class="kanews-form">      
														<select id="post_tags" name="post_tags[]" class="form-control" multiple="multiple"></select>
													</div>
												</div>
											<?php endif; ?>
											
										</div>

										<div id="editor" class="kanews-article-content"></div>
										<textarea name="content" style="display:none" id="content"></textarea>

										<div id="image-preview" class="mt-1" style="position:relative;display: none">
										<div class="kanews-modal-close-btn" ><i class="icon-close"></i></div>
										<img style="width: 100%; height: auto" src="" class="preview-image">
										</div>

										<div class="row">
											<div class="col-12 mt-1">
												
												<input type="file" name="featured" id="featured" accept="image/*" class="kanews-input-file"><label for="featured" class="kanews-input-file-path"><?php esc_html_e( 'Öne Çıkarılan Görsel Ekle', 'kanews' ); ?></label></input>
											</div>
											<div class="col-12">
											<small class="d-block text-muted">
												<?php esc_html_e('Maksimum dosya boyutu: 1 MB', 'kanews'); ?><br>
												<?php esc_html_e('Maksimum çözünürlük: 2000x2000 px', 'kanews'); ?><br>
												<?php esc_html_e('İzin verilen formatlar: JPG, PNG, WebP', 'kanews'); ?>
										</small>
											</div>
											<?php echo kanews_get_recaptcha('registration_form'); ?>
											<div class="col-12">
												<button class="kanews-btn kanews-btn-block" name="post-submit" type="submit"><?php esc_html_e( 'Yayınla', 'kanews' ); ?> <div id="kanews-loader"></div></button>
											</div>
										</div>
									</form>
									
									<script data-no-optimize="1" src="//cdn.quilljs.com/1.3.6/quill.min.js"></script>
									<script>
										// editor
										var toolbarOptions = [
											[{ 'header': [1, 2, 3, 4, 5, 6, false] }],
											['bold', 'italic', 'underline'],
											[{ 'list': 'ordered'}, { 'list': 'bullet' }],
											['link'],
											['image'],
											['clean']                                         // remove formatting button
										];
										var quill = new Quill('#editor', {
										modules: {
											toolbar: toolbarOptions
										},
										theme: 'snow'
										});

										const file = document.getElementById('featured'),
													image = document.querySelector('.preview-image')

										file.addEventListener('change', function() {
											[... this.files].map(file => {
												if (file.name.toLowerCase().match(/\.(jpe?g|png|webp|gif)$/)) {
													const reader = new FileReader()
													reader.addEventListener('load', function () {
														image.src = this.result
													})
													reader.readAsDataURL(file)
													document.getElementById("image-preview").style.display = "block";
												}
											})
										})


										document.querySelector('#image-preview .kanews-modal-close-btn').onclick = function(){
											file.value=''
											image.src=''
											document.getElementById("image-preview").style.display = "none";
										}
									</script>
								<?php else: echo esc_html_e( 'Bu sayfa sizin için devre dışı bırakılmıştır.', 'kanews' ) ?>

								<?php endif; ?>
								<?php else: echo esc_html_e( 'Bu sayfa geçici olarak devre dışı bırakılmıştır.', 'kanews' ); endif; break;
								default: ?>
									<div class="kanews-custom-page-content">
										<h2 class="kanews-custom-page-title"><?php echo sprintf(esc_html__( 'Merhaba, %s', 'kanews' ),
											'<a href="'.get_author_posts_url( get_the_author_meta( 'ID', $current_user->ID ) ).'">'.get_the_author_meta('display_name', $current_user->ID).'</a>'
										); ?></h2>
										<p><?php echo kanews_translate('account-text', esc_html__('Hesap panonuz sayfasında kaydettiğiniz haberlere, yaptığınız yorumlara ulaşabilir ve şifreniz ile hesap ayrıntılarınızı düzenleyebilirsiniz.', 'kanews')); ?></p>

										<?php if(!kanews_user_status($user_id)): ?>
										<?php if(!empty($error)): ?>
											<p class="error error-succ"><?php echo $error; ?></p>
										<?php endif; ?>
										<div class="geo-verification mb-2">
											<h4 style="margin-bottom:1em"><?php esc_html_e('E-Posta Onay İşlemleri', 'kanews'); ?></h4>
											<form method="post" action="<?php the_permalink(); ?>">
												<div class="kanews-form">      
												<input type="number"  min="0" max="999999" onKeyUp="if(this.value>999999){this.value='';}else if(this.value<0){this.value='0';}" id="verification_code" name="verification_code" class="form-control" value="" required>
												<label for="verification_code"><?php _e('Doğrulama Kodunuz', 'kanews'); ?></label>
												</div>
												<input name="verification_submit" type="submit" id="verification_submit" class="kanews-btn kanews-btn-block" value="<?php _e('Hesabımı Doğrula', 'kanews'); ?>" />
											</form>
											<?php if ( false === ( get_transient( 'kanews-user-auth-cache-' . $user_id ) ) ): ?>
												<p><?php echo sprintf(__('Doğrulama kodu almak için <a style="font-weight:bold" href="%s">tıklayın.</a></p>', 'kanews'), esc_url( home_url(add_query_arg( 'verification', true, $_SERVER['REQUEST_URI'] )) ) ); ?>
											<?php else: ?>
												<p><?php esc_html_e('Doğrulama kodu gönderilmiştir, 15 dk içerisinde tekrar talep edebilirsiniz.', 'kanews'); ?></p>
											<?php endif; ?>
										</div>
									<?php endif; ?>
										<div class="row">

											<?php if(kanews_get_option('scoring_system')): ?>
					
												
												<?php 
												$rank_info = kanews_get_rank_info($current_user->ID);
												$rank_badges = kanews_get_option('user-badge-2');
												?>
												<div class="col">
													<div class="kanews-iconbox">
													<div class="kanews-iconbox-icon"><i class="icon-bolt"></i></div>
														<h4 class="kanews-iconbox-title"><?php esc_html_e( 'Puanım', 'kanews' ); ?></h4>
														
														<div class="kanews-iconbox-count"><?php echo get_user_meta( $current_user->ID, 'kan_user_score', true ) ? get_user_meta( $current_user->ID, 'kan_user_score', true ) : '0'; ?></div>

								
													</div>
												</div>
											<?php endif; ?>
											<div class="col">
												<a href="<?php the_permalink();?>?islem=iceriklerim" class="kanews-iconbox">
													<div class="kanews-iconbox-icon"><i class="icon-article"></i></div>
													<h4 class="kanews-iconbox-title"><?php esc_html_e( 'İçeriklerim', 'kanews' ); ?></h4>
													<div class="kanews-iconbox-count"><?php echo count_user_posts( $current_user->ID ); ?></div>
												</a>
											</div>
											<div class="col">
												<a href="<?php the_permalink();?>?islem=yorumlar" class="kanews-iconbox">
													<div class="kanews-iconbox-icon"><i class="icon-comment"></i></div>
													<h4 class="kanews-iconbox-title"><?php esc_html_e( 'Yorumlarım', 'kanews' ); ?></h4>
													<div class="kanews-iconbox-count"><?php echo kanews_get_comments_count_by_user($current_user->ID); ?></div>
												</a>
											</div>
											<?php if(kanews_get_option('post-bookmark')): ?>
											<div class="col">
												<a href="<?php the_permalink();?>?islem=favoriler" class="kanews-iconbox">
													<div class="kanews-iconbox-icon"><i class="icon-bookmark"></i></div>
													<h4 class="kanews-iconbox-title"><?php esc_html_e( 'Favorilerim', 'kanews' ); ?></h4>
													<div class="kanews-iconbox-count"><?php echo get_user_bookmark_count($current_user->ID); ?></div>
												</a>
											</div>
											<?php endif; ?>
											<?php if(kanews_get_option('post-like')): ?>
											<div class="col">
												<a href="<?php the_permalink();?>?islem=begeniler" class="kanews-iconbox">
													<div class="kanews-iconbox-icon"><i class="icon-like"></i></div>
													<h4 class="kanews-iconbox-title"><?php esc_html_e( 'Beğendiklerim', 'kanews' ); ?></h4>
													<div class="kanews-iconbox-count"><?php echo get_user_liked_count($current_user->ID); ?></div>
												</a>
											</div>
											<?php endif; ?>
											<?php if(kanews_get_option('scoring_system')): ?>
					
												
					<?php 
					$rank_info = kanews_get_rank_info($current_user->ID);
					$rank_badges = kanews_get_option('user-badge-2');
					?>
					<div class="col-12 mt-4">
						<div class="kanews-iconbox">


							<div class="kanews-rank-content">

								
								<?php if ($rank_info['next_rank']) : ?>
								<div class="kanews-rank-next">
									<span class="kanews-rank-label"><?php esc_html_e('Sonraki Rütbe:', 'kanews'); ?></span>
									<?php echo kanews_get_rank_badge_html($rank_info['next_rank']); ?>
									
									<div class="kanews-rank-progress">
										<div class="kanews-progress-bar">
											<div class="kanews-progress-fill" style="width: <?php echo esc_attr($rank_info['progress']); ?>%"></div>
										</div>
										<span class="kanews-progress-text">
											<?php printf(
												esc_html__('Sonraki rütbeye %d puan kaldı (%d%%)', 'kanews'),
												$rank_info['next_rank_score'] - $rank_info['current_score'],
												$rank_info['progress']
											); ?>
										</span>
									</div>
								</div>
								<?php endif; ?>
								
								<?php if (is_array($rank_badges) && !empty($rank_badges)) : ?>
								<div class="kanews-rank-all mt-4">
									<h4 class="kanews-rank-section-title"><?php esc_html_e('Tüm Rütbeler', 'kanews'); ?></h4>
									<div class="kanews-rank-badges">
										<?php foreach ($rank_badges as $badge_key => $badge) : 
											$is_current = ($badge_key === $rank_info['current_rank']);
											$is_achieved = ($rank_info['current_score'] >= (int)$badge['min']);
											$badge_class = $is_current ? 'current' : ($is_achieved ? 'achieved' : 'locked');
										?>
										<div class="kanews-rank-badge-item <?php echo esc_attr($badge_class); ?>">
											<?php echo kanews_get_rank_badge_html($badge_key); ?>
											<div class="kanews-rank-badge-info">
												<span class="kanews-rank-badge-name"><?php echo esc_html($badge['title']); ?></span>
												<span class="kanews-rank-badge-score">
													<?php printf(
														esc_html__('%d - %d puan', 'kanews'),
														(int)$badge['min'],
														(int)$badge['max']
													); ?>
												</span>
											</div>
										</div>
										<?php endforeach; ?>
									</div>
								</div>
								<?php endif; ?>
							</div>
						</div>
					</div>
				<?php endif; ?>
											
										</div>
									</div>
								<?php
							} ?>

						</div>
					</div>

				</div>
			</div>
		</main>

		<div id="kanews-upload-user-avatar" class="kanews-modal">
			<div data-toggle="kanews-upload-user-avatar" class="kanews-modal-close"></div>
			<div class="kanews-modal-content">
				<div class="kanews-modal-title"><?php esc_html_e( 'Profil Görsellerini Yükle', 'kanews' ); ?></div>
				<div data-toggle="kanews-upload-user-avatar" class="kanews-modal-close-btn"><i class="icon-close"></i></div>
				<div class="kanews-modal-content-inner">
					<form method="post" id="your-profile" action="<?php the_permalink(); ?>" enctype="multipart/form-data">
					<input name="updateuser" type="submit" id="updateuser" class="kanews-btn kanews-btn-block bg-success" value="<?php _e('Değişiklikleri Kaydet', 'kanews'); ?>" />

						<?php $false = 'off'; do_action('edit_user_profile',$current_user, $false); ?>
						<?php wp_nonce_field( 'update-user_'. $current_user->ID ); ?>
						<input name="action" type="hidden" id="action" value="update-user" />
					</form>
				</div>
			</div>
		</div>
	<?php else: ?>

		<?php include( KAN_TEMPLATE_PATH.'/inc/dashboard/dashboard-login.php'); ?>

	<?php endif; ?>
<?php endwhile; endif; ?>
<?php if(is_user_logged_in()): ?>
<?php get_footer(); ?>
<?php else: ?>
<?php wp_footer(); ?>
<?php endif; ?>

<script data-no-optimize="1" src="//ajax.googleapis.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
<link data-no-optimize="1" href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
<script data-no-optimize="1" src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>

<script data-no-optimize="1">
$('#post_tags').select2({
	tags: true,
	tokenSeparators: [','], 
	placeholder: "<?php esc_html_e('Etiket girin', 'kanews'); ?>",
	selectOnClose: true, 
	maximumSelectionLength: 10,
	closeOnSelect: false,
	"language":{
		"noResults" : function () { return '<?php esc_html_e("Eklemek için bir şeyler yazın", "kanews"); ?>'; }
	}
});

$('#cat').select2({
	placeholder: "<?php esc_html_e('Kategori seçin', 'kanews'); ?>",
	maximumSelectionLength: 3
});

</script>
<script>

$("#post-submit").on("submit",function() {
  $("#content").val($("#editor .ql-editor").html());
})



</script>