<?php
defined( 'ABSPATH' ) || exit; // Exit if accessed directly

/* ------------------------------------------------------------------------- *
	* Loader
/* ------------------------------------------------------------------------- */
add_action( 'init', 'kanews_add_shortcodes' );
function kanews_add_shortcodes() {
	$shortcodes = array(
		'calculator',
		'btn',
		'alert',
		'accordion',
		'collapse',
		// 'authors',
		'dropcap',
		'highlight',
		'related-posts',
		'related-post',
		'title',
		'read-more',
		'prayer',
		'currencies',
		'exchange',
		'sources',
		'source',
		'date-today',
		'timeline',
		'weather'
	);

	foreach ( $shortcodes as $shortcode ) {
		add_shortcode( 'kanews-' . $shortcode, 'kanews_shortcode_' . str_replace( '-', '_', $shortcode ) );
	}
}

// Enqueue calculator scripts
function kanews_enqueue_calculator_scripts() {
    // Only enqueue if the calculator shortcode is used on the page
    global $post;
    if (is_a($post, 'WP_Post') && has_shortcode($post->post_content, 'kanews-calculator')) {
        wp_enqueue_script('kanews-calculator');
    }
}
add_action('wp_enqueue_scripts', 'kanews_enqueue_calculator_scripts');

// Add AJAX handler for calculator loading
add_action('wp_ajax_kanews_load_calculator', 'kanews_ajax_load_calculator');
add_action('wp_ajax_nopriv_kanews_load_calculator', 'kanews_ajax_load_calculator');

function kanews_ajax_load_calculator() {
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'kanews-ajax-nonce')) {
        wp_send_json_error('Invalid nonce');
        return;
    }

    // Get calculator type
    $calculator_type = isset($_POST['calculator']) ? sanitize_text_field($_POST['calculator']) : '';
    
    if (empty($calculator_type)) {
        wp_send_json_error('Calculator type is required');
        return;
    }

    // Get calculator HTML based on type
    $calculator_html = '';
    switch ($calculator_type) {
        case 'doviz':
            $calculator_html = render_currency_calculator();
            break;
        case 'coin':
            $calculator_html = render_crypto_calculator();
            break;
        case 'emtia':
            $calculator_html = render_commodity_calculator();
            break;
        case 'burc':
            $calculator_html = render_horoscope_calculator();
            break;
        case 'vki':
            $calculator_html = render_bmi_calculator();
            break;
        case 'kredi':
            $calculator_html = render_credit_calculator();
            break;
        case 'birikim':
            $calculator_html = render_savings_calculator();
            break;
        case 'vergi':
            $calculator_html = render_tax_calculator();
            break;
        default:
            wp_send_json_error('Invalid calculator type');
            return;
    }

    wp_send_json_success(array('html' => $calculator_html));
}

/* ------------------------------------------------------------------------- *
	* Helpers
/* ------------------------------------------------------------------------- */
function parse_data_attributes( $data ) {
	$result = array();
	
	// If $data is already an array, return it directly
	if (is_array($data)) {
		return $data;
	}
	
	// If $data is a string, parse it
	if (is_string($data)) {
		$pairs = explode('|', $data);
		
		foreach ($pairs as $pair) {
			$item = explode('=', $pair);
			if (count($item) == 2) {
				$result[trim($item[0])] = trim($item[1]);
			}
		}
	}
	
	return $result;
}

/* ------------------------------------------------------------------------- *
	* 1.0 Calculator
/* ------------------------------------------------------------------------- */
function kanews_shortcode_calculator($attrs = []) {
	// Parse attributes
	$attrs = parse_data_attributes($attrs);
	
	// Default calculator type
	$default_type = isset($attrs['tur']) ? sanitize_text_field($attrs['tur']) : 'doviz';
	
	// Check if tabs should be shown (hidden by default)
	$show_tabs = isset($attrs['show_tabs']) && ($attrs['show_tabs'] === 'true' || $attrs['show_tabs'] === '1');
	
	// Get excluded calculator types
	$exclude_types = [];
	if (isset($attrs['exclude_types']) && !empty($attrs['exclude_types'])) {
		$exclude_types = explode(',', $attrs['exclude_types']);
		$exclude_types = array_map('trim', $exclude_types);
	}
	
	// Get index parameter for crypto calculator
	$index = isset($attrs['index']) ? sanitize_text_field($attrs['index']) : '';
	
	// Valid calculator types
	$calculator_types = array(
		'doviz' => 'render_currency_calculator',
		'coin' => 'render_crypto_calculator',
		'emtia' => 'render_commodity_calculator',
		'burc' => 'render_horoscope_calculator',
		'vki' => 'render_bmi_calculator',
		'kredi' => 'render_credit_calculator',
		'birikim' => 'render_savings_calculator',
	);
	
	// Remove excluded calculator types
	foreach ($exclude_types as $exclude_type) {
		if (isset($calculator_types[$exclude_type])) {
			unset($calculator_types[$exclude_type]);
		}
	}
	
	// Check if the requested calculator type exists
	if (!isset($calculator_types[$default_type])) {
		$default_type = 'doviz'; // Fallback to default if invalid type
	}
	
	// Start output buffering
	ob_start();
	?>
	<div class="kanews-calculator-wrapper">
		<?php if ($show_tabs && !empty($calculator_types)): ?>
		<div class="kanews-tab-2">
		<div class="nav nav-tabs d-flex flex-wrap p-1">
			<?php foreach ($calculator_types as $type => $render_function): ?>
				<div class="nav-item">
				<button class="nav-link calculator-tab <?php echo $type === $default_type ? 'active' : ''; ?>" 
						data-calculator="<?php echo esc_attr($type); ?>">
					<?php 
					$labels = array(
						'doviz' => __('Döviz', 'kanews'),
						'coin' => __('Kripto', 'kanews'),
						'emtia' => __('Emtia', 'kanews'),
						'burc' => __('Burç', 'kanews'),
						'vki' => __('VKİ', 'kanews'),
						'kredi' => __('Kredi', 'kanews'),
						'birikim' => __('Birikim', 'kanews')
					);
					echo esc_html($labels[$type] ?? ucfirst($type)); 
					?>
				</button>
				</div>
			<?php endforeach; ?>
		</div>
		</div>
		<?php endif; ?>
		<div class="kanews-calculator-container">
			<?php 
			$render_function = $calculator_types[$default_type];
			if (function_exists($render_function)) {
				if ($default_type === 'coin' && !empty($index)) {
					echo call_user_func($render_function, $index);
				} else {
					echo call_user_func($render_function);
				}
			}
			?>
		</div>
	</div>
	<?php
	return ob_get_clean();
}

function render_currency_calculator($index = '') {
    $response = get_local_file_contents(KAN_TEMPLATE_PATH."/inc/api/currencies/currencies.json");
          $data = json_decode($response, true);

    if(isset($data['data'])) {
            $data = $data['data'];
          }

    if(empty($data) || !is_array($data) || !isset($data['USD'])) {
        return '';
    }

    ob_start(); ?>
    <form id="doviz_calc" class="kanews-calculator">
        <div class="row position-relative">
            <div class="col-12 col-md-6">
                <div class="kanews-form">
                    <select class="currency_switcher_x" aria-label="<?php esc_html_e('Miktar', 'kanews'); ?>">
                        <?php foreach($data as $name => $currency): ?>
                            <option value="<?php echo esc_attr($currency['Selling']); ?>" 
                                <?php selected(isset($_GET['parite']) && strtoupper($_GET['parite']) == $name); ?>>
                                <?php echo esc_html($name); ?>
                            </option>
                        <?php endforeach; ?>
                        <option value="1">TRY</option>
                    </select>
                    <label><?php _e('Miktar', 'kanews'); ?></label>
                </div>
            </div>
            <div class="exchange-icon hidden-mobile">
                <svg aria-hidden="true" focusable="false" role="none" width="24" height="24" fill="currentColor" viewBox="0 0 24 24">
                    <path fill-rule="evenodd" d="m16.629 11.999-1.2-1.2 3.085-3.086H2.572V5.999h15.942L15.43 2.913l1.2-1.2 4.543 4.543a.829.829 0 0 1 0 1.2l-4.543 4.543Zm-9.257-.001 1.2 1.2-3.086 3.086h15.943v1.714H5.486l3.086 3.086-1.2 1.2-4.543-4.543a.829.829 0 0 1 0-1.2l4.543-4.543Z" clip-rule="evenodd">
                    </path>
                </svg>
            </div>
            <div class="col-12 col-md-6">
                <div class="kanews-form">
                    <select class="currency_switcher_2" aria-label="<?php esc_html_e('Dönüştürülen', 'kanews'); ?>">
                        <option value="1">TRY</option>
                        <?php foreach($data as $name => $currency): ?>
                            <option value="<?php echo esc_attr($currency['Selling']); ?>">
                                <?php echo esc_html($name); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <label><?php _e('Dönüştürülen', 'kanews'); ?></label>
                </div>
            </div>
            <div class="col-12">
                <div class="kanews-form">
                    <input type="text" class="currency3value" value="1" aria-label="<?php esc_html_e('Miktar', 'kanews'); ?>">
                    <label><?php _e('Miktar', 'kanews'); ?></label>
                </div>
            </div>
            <div class="col-12">
                <div class="kanews-form">
                    <input type="text" class="currency4value" value="?" aria-label="<?php esc_html_e('Sonuç', 'kanews'); ?>" readonly>
                    <label><?php _e('Sonuç', 'kanews'); ?></label>
                </div>
            </div>
        </div>
        <div class="calculator-info">
            <div class="info-row">
                <span class="info-label"><?php _e('Çeviri Bilgisi', 'kanews'); ?></span>
                <span class="info-value conversion-info">-</span>
            </div>
            <div class="info-row">
                <span class="info-label"><?php _e('Birim Değeri', 'kanews'); ?></span>
                <span class="info-value selected-value">0.00</span>
            </div>
        </div>
    </form>
    <?php
    return ob_get_clean();
}

function render_crypto_calculator($index = '') {
      $index = !empty($index) ? $index : kanews_get_option('crypto-page-index', 'try');
      $data = json_decode(kanews_crypto_data($index)); 
      
    if(empty($data)) {
        return '';
    }

    ob_start(); ?>
    <form id="coin_calc" class="kanews-calculator">
        <div class="row position-relative">
            <div class="col-12 col-md-6">
                <div class="kanews-form">
                    <select class="currency_switcher_x" aria-label="<?php esc_html_e('Para Birimi', 'kanews'); ?>">
                        <?php foreach($data as $coin): ?>
                            <option value="<?php echo esc_attr($coin->current_price); ?>">
                                <?php echo esc_html(strtoupper($coin->symbol)); ?>
                            </option>
                        <?php endforeach; ?>
                        <option value="1"><?php echo esc_html(strtoupper($index)); ?></option>
                    </select>
                    <label><?php _e('Para Birimi', 'kanews'); ?></label>
            </div>
          </div>
            <div class="exchange-icon hidden-mobile">
                <svg aria-hidden="true" focusable="false" role="none" width="24" height="24" fill="currentColor" viewBox="0 0 24 24">
                    <path fill-rule="evenodd" d="m16.629 11.999-1.2-1.2 3.085-3.086H2.572V5.999h15.942L15.43 2.913l1.2-1.2 4.543 4.543a.829.829 0 0 1 0 1.2l-4.543 4.543Zm-9.257-.001 1.2 1.2-3.086 3.086h15.943v1.714H5.486l3.086 3.086-1.2 1.2-4.543-4.543a.829.829 0 0 1 0-1.2l4.543-4.543Z" clip-rule="evenodd">
                    </path>
                </svg>
            </div>
        <div class="col-12 col-md-6">
                <div class="kanews-form">
                    <select class="currency_switcher_3" aria-label="<?php esc_html_e('Dönüştürülen', 'kanews'); ?>">
                        <option value="1"><?php echo esc_html(strtoupper($index)); ?></option>
                        <?php foreach($data as $coin): ?>
                            <option value="<?php echo esc_attr($coin->current_price); ?>">
                                <?php echo esc_html(strtoupper($coin->symbol)); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <label><?php _e('Dönüştürülen', 'kanews'); ?></label>
          </div>
        </div>
            <div class="col-12">
                <div class="kanews-form">
                    <input type="text" class="currency5value" value="1" aria-label="<?php esc_html_e('Miktar', 'kanews'); ?>">
                    <label><?php _e('Miktar', 'kanews'); ?></label>
          </div>
        </div>
        <div class="col-12">
                <div class="kanews-form">
                    <input type="text" class="currency6value" value="?" aria-label="<?php esc_html_e('Sonuç', 'kanews'); ?>" readonly>
                    <label><?php _e('Sonuç', 'kanews'); ?></label>
          </div>
        </div>
        </div>
        <div class="calculator-info">
            <div class="info-row">
                <span class="info-label"><?php _e('Çeviri Bilgisi', 'kanews'); ?></span>
                <span class="info-value conversion-info">-</span>
            </div>
            <div class="info-row">
                <span class="info-label"><?php _e('Birim Değeri', 'kanews'); ?></span>
                <span class="info-value selected-value">0.00</span>
        </div>
      </div>
    </form>
    <?php
    return ob_get_clean();
}

function render_bmi_calculator() {
    ob_start(); ?>
    <form name="BedenKitleEndeksi" class="kanews-calculator">
        <div class="row">
            <div class="col-12 col-md-6">
          <div class="kanews-form">
                    <input type="text" id="kilosu" placeholder="<?php esc_html_e('Kilonuzu giriniz', 'kanews'); ?>">
                    <label><?php _e('Kilo (kg)', 'kanews'); ?></label>
          </div>
          </div>
            <div class="col-12 col-md-6">
          <div class="kanews-form">
                    <input type="text" id="boyu" placeholder="<?php esc_html_e('Boyunuzu giriniz', 'kanews'); ?>">
                    <label><?php _e('Boy (cm)', 'kanews'); ?></label>
          </div>
          </div>
            <div class="col-12">
                <button type="button" class="kanews-btn kanews-btn-block kanews-btn-sm mt-0"><?php _e('Hesapla', 'kanews'); ?></button>
          </div>
            <div class="col-12">
                <button type="reset" class="kanews-btn kanews-btn-light kanews-btn-block mt-0 kanews-btn-sm"><?php _e('Sıfırla', 'kanews'); ?></button>
          </div>
        </div>
        <div class="calculator-info">
            <div class="info-row">
                <span class="info-label"><?php _e('Vücut Kitle İndeksi', 'kanews'); ?></span>
                <span class="info-value bmi-value">-</span>
            </div>
            <div class="info-row">
                <span class="info-label"><?php _e('Sonuç', 'kanews'); ?></span>
                <span class="info-value bmi-result">-</span>
            </div>
          </div>
        </form>
    <?php
    return ob_get_clean();
}

function render_commodity_calculator() {
    $response = get_local_file_contents(KAN_TEMPLATE_PATH."/inc/api/gold/gold.json");
    $data = json_decode($response, true);

    if(isset($data['data'])) {
        $data = $data['data'];
    }

    if(empty($data) || !is_array($data)) {
        return '';
    }

    $response2 = get_local_file_contents(KAN_TEMPLATE_PATH."/inc/api/currencies/currencies.json");
    $data2 = json_decode($response2, true);

    if(isset($data2['data'])) {
        $data2 = $data2['data'];
    }

    if(empty($data2) || !is_array($data2) || !isset($data2['USD'])) {
        return '';
    }

    

    ob_start(); ?>
    <form id="emtia_calc" class="kanews-calculator">
        <div class="row position-relative">
            <div class="col-12 col-md-6">
                <div class="kanews-form">
                    <select class="currency_switcher_x" aria-label="<?php esc_html_e('Para Birimi', 'kanews'); ?>">
                        <?php foreach($data as $name => $commodity): ?>
                            <option value="<?php echo esc_attr($commodity['Selling']); ?>" 
                                <?php selected(isset($_GET['emtia']) && strtoupper($_GET['emtia']) == $name); ?>>
                                <?php echo esc_html(kanews_get_emtia_name($name)); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <label><?php _e('Para Birimi', 'kanews'); ?></label>
          </div>
        </div>
            <div class="exchange-icon hidden-mobile">
                <svg aria-hidden="true" focusable="false" role="none" width="24" height="24" fill="currentColor" viewBox="0 0 24 24">
                    <path fill-rule="evenodd" d="m16.629 11.999-1.2-1.2 3.085-3.086H2.572V5.999h15.942L15.43 2.913l1.2-1.2 4.543 4.543a.829.829 0 0 1 0 1.2l-4.543 4.543Zm-9.257-.001 1.2 1.2-3.086 3.086h15.943v1.714H5.486l3.086 3.086-1.2 1.2-4.543-4.543a.829.829 0 0 1 0-1.2l4.543-4.543Z" clip-rule="evenodd">
                    </path>
                </svg>
            </div>
            <div class="col-12 col-md-6">
                <div class="kanews-form">
                    <select class="currency_switcher_2" aria-label="<?php esc_html_e('Dönüştürülen', 'kanews'); ?>">
                        <option value="1">TRY</option>
                        <?php foreach($data2 as $name => $currency): ?>
                            <?php if(isset($currency['Name'])): ?>
                                <option value="<?php echo esc_attr($currency['Selling']); ?>">
                                    <?php echo esc_html($currency['Name']); ?>
                                </option>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </select>
                    <label><?php _e('Dönüştürülen', 'kanews'); ?></label>
                </div>
            </div>
        <div class="col-12">
                <div class="kanews-form">
                    <input type="text" class="currency3value" value="1" aria-label="<?php esc_html_e('Miktar', 'kanews'); ?>">
                    <label><?php _e('Miktar', 'kanews'); ?></label>
          </div>
            </div>
            <div class="col-12">
                <div class="kanews-form">
                    <input type="text" class="currency4value" value="?" aria-label="<?php esc_html_e('Sonuç', 'kanews'); ?>" readonly>
                    <label><?php _e('Sonuç', 'kanews'); ?></label>
                </div>
            </div>
        </div>
        <div class="calculator-info">
            <div class="info-row">
                <span class="info-label"><?php _e('Çeviri Bilgisi', 'kanews'); ?></span>
                <span class="info-value conversion-info">-</span>
            </div>
            <div class="info-row">
                <span class="info-label"><?php _e('Birim Değeri', 'kanews'); ?></span>
                <span class="info-value selected-value">0.00</span>
            </div>
        </div>
    </form>
    <?php
    return ob_get_clean();
}

/* ------------------------------------------------------------------------- *
	* 2.0 Button
/* ------------------------------------------------------------------------- */
function kanews_shortcode_btn( $atts, $content = null ) {
	$size  = 'kanews-btn-sm';
	$link = $style = $color = $button_target  =  '';
	
  if( is_array( $atts ) ) extract($atts);

	if( !empty( $target ) && $target == true ) $button_target = ' target="_blank"';
	if( !empty( $icon ) )   $icon   = '<i class="fa '.$icon.'"></i>';
	
  $out = '<a href="'.$link.'"'.$button_target.' class="kanews-btn '.$style.' '.$size.' '.$color.'">'. do_shortcode($content). '</a>';
  
  return $out;
}

/* ------------------------------------------------------------------------- *
	* 3.0 Alert
/* ------------------------------------------------------------------------- */
function kanews_shortcode_alert( $atts, $content = null ) {
	$color = 'bg-success';
  if( is_array( $atts ) ) extract($atts);
	$out = '<div role="alert" class="kanews-response '.$color.'">'.do_shortcode($content). '</div>';
  return $out;
}

/* ------------------------------------------------------------------------- *
	* 4.0 Accordion
/* ------------------------------------------------------------------------- */
function kanews_shortcode_accordion( $atts , $content = null ) {

		if ( isset( $GLOBALS['collapsibles_count'] ) ) {
			$GLOBALS['collapsibles_count'] ++;
		} else {
			$GLOBALS['collapsibles_count'] = 0;
		}

		$atts = shortcode_atts( array(
			"xclass" => false,
			"data"   => false
		), $atts );

		$class = 'accordion';
		$class .= ( $atts['xclass'] ) ? ' ' . $atts['xclass'] : '';

		$id = 'custom-collapse-' . $GLOBALS['collapsibles_count'];

		$data_props = parse_data_attributes( $atts['data'] );

		return sprintf(
			'<div class="%s" id="%s" role="tablist" aria-multiselectable="true"%s>%s</div>',
			esc_attr( trim( $class ) ),
			esc_attr( $id ),
			( $data_props ) ? ' ' . $data_props : '',
			do_shortcode( $content )
		);


}

/* ------------------------------------------------------------------------- *
	* 5.0 Collapse
/* ------------------------------------------------------------------------- */
function kanews_shortcode_collapse( $atts , $content = null ) {

  if ( isset( $GLOBALS['single_collapse_count'] ) ) {
    $GLOBALS['single_collapse_count'] ++;
  } else {
    $GLOBALS['single_collapse_count'] = 0;
  }

  $atts = shortcode_atts( array(
    "title"  => false,
    "type"   => false,
    "active" => false,
    "xclass" => false,
    "data"   => false
  ), $atts );

  $card_class = 'collapse-item';
  $card_class .= ( $atts['xclass'] ) ? ' ' . $atts['xclass'] : '';

  $collapse_class = 'collapse';
  $collapse_class .= ( $atts['active'] == 'true' ) ? ' in' : '';

  $a_class = 'collapse-btn ';
  $a_class .= ( $atts['active'] == 'true' ) ? '' : 'collapsed';

  $parent           = isset( $GLOBALS['collapsibles_count'] ) ? 'custom-collapse-' . $GLOBALS['collapsibles_count'] : 'single-collapse';
  $current_collapse = $parent . '-' . $GLOBALS['single_collapse_count'];

  $data_props = parse_data_attributes( $atts['data'] );

  return sprintf(
    '<div class="%1$s"%2$s>
    <div class="collapse-item-header">
      <button class="%3$s" data-toggle="collapse"%4$s href="#%5$s">
        <h3>
          %6$s
        </h3>
      </button>
      </div>
      <div id="%5$s" class="%7$s">
        <div class="collapse-item-body">%8$s</div>
      </div>
    </div>',
    esc_attr( $card_class ),
    ( $data_props ) ? ' ' . $data_props : '',
    $a_class,
    ( $parent ) ? ' data-parent="#' . $parent . '"' : '',
    $current_collapse,
    $atts['title'],
    esc_attr( $collapse_class ),
    do_shortcode( $content )
  );
}

/* ------------------------------------------------------------------------- *
	* 6.0 Weather
/* ------------------------------------------------------------------------- */
function kanews_shortcode_weather( $atts, $content = null ) {

  $atts = shortcode_atts( array(
    'page'     => 1,
    'city'     => 'İstanbul',
    'bg_color' => ''
  ), $atts );

  ob_start();
  new KANEWS_WEATHER( $atts );

  return ob_get_clean();
}

/* ------------------------------------------------------------------------- *
	* 6.0 Alert
/* ------------------------------------------------------------------------- */
function kanews_shortcode_dropcap( $atts, $content = null ) {
	$type = '';
	if( is_array( $atts ) ) extract($atts);
  return '<span class="kanews-text-dropcap ' . $color . '">' . $content . '</span>';
}

/* ------------------------------------------------------------------------- *
	* 7.0 Highlight
/* ------------------------------------------------------------------------- */
function kanews_shortcode_highlight( $atts, $content = null ) {
  return '<span class="kanews-text-highlight">' . $content . '</span>';
}

/* ------------------------------------------------------------------------- *
	* 8.0 Related Posts
/* ------------------------------------------------------------------------- */
function kanews_shortcode_related_posts( $atts, $content = null ) {
  $title = kanews_translate('related-posts-text', esc_html__( 'Benzer Haberler', 'kanews' ));
  $style = '1';
  $tag = 'h4';
  $p = '';
  $pids = '';
  if( is_array( $atts ) ) extract($atts);

  $pids =  explode(',', $p);
  $kanews_query = new WP_Query(
    array( 'post__in'=> $pids)
  );

  switch($style) {
    case '2':
      $classes = 'kanews-post-grid-item';
      $col = 'col-6 col-md-4';
      $thumbnail = 'kanews-post-2';
    break;
    default:
      $classes = 'kanews-post-grid-item kanews-post-small-item';
      $col = 'col-12 col-md-6';
      $thumbnail = 'thumbnail';
    break;
  }

  $out = '';

  if ($kanews_query->have_posts()) {
    $out .= '<div class="kanews-posts-shortcode mb-1">';
      $out .= kanews_widget_title($title, 'div');
      $out .= '<div class="row">';
      while($kanews_query->have_posts()) : $kanews_query->the_post(); global $post;
        $default = kanews_get_option('default-thumbnail') ? kanews_get_option('default-thumbnail') : '';

        $out .= '<div class="'.$col.'">';
          $out .= '<div class="kanews-post-item '.$classes.'">';
            $out .= '<a href="' . esc_url(get_permalink()) . '" class="kanews-post-href" aria-label="'.get_the_title().'"></a>';
            $out .= kanews_post_thumbnail($thumbnail, 'kanews-post-thumb', false, true);
            $out .= '<div class="kanews-post-content">';
              // $out .= kanews_primary_category(get_the_ID());
              $out .= '<'.$tag.' class="kanews-post-headline truncate truncate-2">' . get_the_title() . '</'.$tag.'>';
            $out .= '</div>';
          $out .= '</div>';
        $out .= '</div>';
      endwhile;
      wp_reset_postdata();
      $out .= '</div>';
    $out .= '</div>';
  } else {
    return false;
  }

  return $out;
}
/* ------------------------------------------------------------------------- *
	* 8.0 Related Posts
/* ------------------------------------------------------------------------- */
function kanews_shortcode_related_post( $atts, $content = null ) {
  $atts = shortcode_atts( array(
    'title'  =>  kanews_translate('related-post-text', esc_html__( 'Gözden Kaçmasın', 'kanews' )),
    'tag'    => 'h4',
    'ids'    => ''
  ), $atts );

  if( is_array( $atts ) ) extract($atts);

  $pids =  explode(',', $ids);
  $kanews_query = new WP_Query(
    array( 'p'=> $ids)
  );

  if(!$tag)
  $tag = 'h4';

  $classes = 'kanews-post-item';
  $col = 'col-12';
  $thumbnail = 'kanews-post-1';

  $out = '';

  if ($kanews_query->have_posts()) {
    $out .= '<div class="kanews-post-shortcode">';
      while($kanews_query->have_posts()) : $kanews_query->the_post(); global $post;
        $default = kanews_get_option('default-thumbnail') ? kanews_get_option('default-thumbnail') : '';
        if (has_post_thumbnail() ) {
          $image_id = get_post_thumbnail_id($post->ID);  
          $image_url = wp_get_attachment_image_src($image_id, $thumbnail);  
          $image_url = isset($image_url[0]) ? $image_url[0] : ''; 
        } elseif($default != '') {
          $image_url = wp_get_attachment_image_src($default['id'], $thumbnail);  
          $image_url = isset($image_url[0]) ? $image_url[0] : '';  
        }
        $out .= '<div class="kanews-post-shortcode-title"><i class="icon-bolt"></i> '.$title.'</div>';
        $out .= '<div class="mt-1 kanews-post-item d-flex flex-wrap '.$classes.'">';
          $out .= '<a href="' . esc_url(get_permalink()) . '" class="kanews-post-href" aria-label="'.get_the_title().'"></a>';
          if( $image_url != '' ) {
            $out .= '<div class="kanews-post-thumb">';
              $out .= '<img alt="thumbnail" loading="lazy" height="90" width="215" src="'.esc_url($image_url).'" />';
            $out .= '</div>';
          }
          $out .= '<div class="kanews-post-content">';
            $out .= '<'.$tag.' class="kanews-post-headline truncate truncate-2">' . get_the_title() . '</'.$tag.'>';
            $out .= '<span>'.esc_html__( 'Haberi görüntüle', 'kanews' ).' <i class="icon-arrow-long icon-180deg"></i> </span>';
          $out .= '</div>';
        $out .= '</div>';
      endwhile;
      wp_reset_postdata();
    $out .= '</div>';
  } else {
    return false;
  }

  return $out;
}

/* ------------------------------------------------------------------------- *
	* 9.0 Title
/* ------------------------------------------------------------------------- */
function kanews_shortcode_title( $atts, $content = null ) {
  if( is_array( $atts ) ) extract($atts);

  $tag = !empty($tag) ? $tag : 'h3';
  $content = !empty($title) ? $title : $content;

  return '<div class="kanews-section-flat">'.kanews_widget_title($content, $tag).'</div>';
}

/* ------------------------------------------------------------------------- *
	* 11.0 Read More
/* ------------------------------------------------------------------------- */
function kanews_shortcode_read_more( $atts, $content = null ) {
  extract(shortcode_atts(array(
		'more' => 'READ MORE',
		'less' => 'READ LESS'
	), $atts));

	mt_srand((double)microtime() * 1000000);
	$rnum = mt_rand();

  $amp2 = 'style="display: none;"';
  $amp = '';
  $onclick = '';

  if(function_exists( 'is_amp_endpoint' ) && is_amp_endpoint()) {
    $amp = 'on="tap:read' . $rnum . '.toggleVisibility"';
    $amp2 = 'hidden';
  } else {
    $onclick = 'onclick="read_toggle(' . $rnum . ', \'' . esc_html__('Devamını Göster', 'kanews') . '\', \'' . esc_html__('Gizle', 'kanews') . '\'); return false;"';
  }
  
	$new_string = '<div class="mt-1 mb-1"><a '.$onclick.' class="kanews-label" id="readlink' . $rnum . '" '.$amp.'">' . esc_html__('Devamını Göster', 'kanews') . '</a></div>' . "\n";
	$new_string .= '<div class="read_div" id="read' . $rnum . '" '.$amp2.'>' . wpautop( do_shortcode($content)) . '</div>';

	return $new_string;
}

/* ------------------------------------------------------------------------- *
	* 12.0 Prayer
/* ------------------------------------------------------------------------- */
function kanews_shortcode_prayer( $atts, $content = null ) {
  $atts = shortcode_atts( array(
    "city" => '',
    "region" => '',
    "class"   => ''
  ), $atts );

  $region = isset($atts['city']) && $atts['city'] != '' ? $atts['city'] : '';
  $city = $atts['region'] != '' ? $atts['region'] : $region;


  $date = date('Y-m-d');
  $data = kanews_prayer_times_api( 'Turkey', $region, $city, $date, 30);

  if( empty($data) || !isset( $data['times'] ) || !isset($data['times'][$date]) )
  return;

  $out = '';

  if(!empty($data) && isset( $data['times'] ) ) {
    $title = wp_sprintf( esc_html__('%1$s Namaz Vakitleri', 'kanews'), $city );
    $out .= kanews_widget_title($title, 'div');
    $out .= '<table class="kanews-table table-currencies" style="table-layout: auto;">';
    $out .= '<thead>';
    $out .= ' <tr>';
    $out .= '<th><span class="nobr">'. esc_html__( 'Tarih', 'kanews' ) .'</span></th>';
    $out .= '<th><span class="nobr">'. esc_html__( 'İmsak', 'kanews' ) .'</span></th>';
    $out .= '<th><span class="nobr">'. esc_html__( 'Güneş', 'kanews' ) .'</span></th>';
    $out .= '<th><span class="nobr">'. esc_html__( 'Öğle', 'kanews' ) .'</span></th>';
    $out .= '<th><span class="nobr">'. esc_html__( 'İkindi', 'kanews' ).'</span></th>';
    $out .= '<th><span class="nobr">'. esc_html__( 'Akşam', 'kanews' ).'</span></th>';
    $out .= '<th><span class="nobr">'. esc_html__( 'Yatsı', 'kanews' ) .'</span></th>';
    $out .= '</tr>';
    $out .= '</thead>';
    $out .= '<tbody>';
    foreach($data['times'] as $key=>$value) {
      echo '<tr class="'.(( $key == $date ) ? 'active' : '' ).'">';
      $tarih_objesi = new DateTime($key);
      $gun = $tarih_objesi->format('D');
      $ay = $tarih_objesi->format('M');
      $yil = $tarih_objesi->format('Y');
      $tarih_turkce = $tarih_objesi->format('d') . ' ' . kanews_translate_months($ay) . ' ' . $tarih_objesi->format('Y') . ' ' . kanews_translate_days($gun);

      $out .= '<td style="font-weight: 600">'.$tarih_turkce.'</td>';
        foreach( $value as $time ) {
          $out .= '<td>'.$time.'</td>';
        }
      $out .= '</tr>';

    }
    $out .= '</tbody></table>';
  } else {
    $out .= '<tr><td>'.esc_html__( ('Veri alınamadı!'), 'kanews' ).'</td></tr>';
  }

	return $out;
}

/* ------------------------------------------------------------------------- *
	* 13.0 Currencies
/* ------------------------------------------------------------------------- */
function kanews_shortcode_currencies( $atts, $content = null ) {
  $data = Kanews\Currencies::update();
  $out = '';

  if( ! isset(  $atts['currencies'] ) ) {
    $currencies = array();
    foreach ($atts as $key => $value) {
      if($value == 'true') {
        $currencies[] = $key;
      }
    }
  } else {
    $currencies = explode( ',', $atts['currencies']);
  }

  // print_r($currencies);

  $out .= '<div class="kanews-section"><div class="kanews-inline-card">';
  foreach($currencies as $currency) {
    if($data->$currency->value == 0)
    $data->$currency->value = number_format((float)$data->$currency->shutdown, 3,",",".");
    $icon = '<svg class="icon-trending" xmlns="http://www.w3.org/2000/svg" height="16" width="18" fill="currentColor"><path d="M3.062 15 2 13.938l5.854-5.855 3.167 3.167 4.417-4.396H13v-1.5h5v5h-1.5V7.917l-5.479 5.458-3.167-3.167Z"/></svg>';
    $index = 'try';
    if ($data->$currency->change_percent < 0)  {
      $class = 'border-danger color-danger';
      $icon = '<svg class="icon-trending icon-180deg" xmlns="http://www.w3.org/2000/svg" height="16" width="18" fill="currentColor"><path d="M3.062 15 2 13.938l5.854-5.855 3.167 3.167 4.417-4.396H13v-1.5h5v5h-1.5V7.917l-5.479 5.458-3.167-3.167Z"/></svg>';
    } else 
      $class = 'border-success color-success';
    if($currency == 'usd') {
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/US.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
    }elseif($currency == 'euro'){
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/EU.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
    }elseif($currency == 'gbp'){
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/GB.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
    }elseif($currency == 'cny'){
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/CN.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
    }elseif($currency == 'jpy'){
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/JP.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
    }elseif($currency == 'rub'){
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/RU.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
    }elseif($currency == 'eur-usd'){
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/EU.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
    }elseif($currency == 'ise-100'){
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'bist-100', 'width' => 18, 'height' => 18), '1e1');
      $img2 = '';
    }elseif($currency == 'gau'){
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/metal/gold.svg', array('alt' => 'gau', 'width' => 18, 'height' => 18), '1e1');
      $img2 = '';
    }elseif($currency == 'btc'){
      $index = kanews_get_option('crypto-page-index', 'try');
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/crypto/XTVCBTC.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
    }elseif($currency == 'eth'){
      $index = kanews_get_option('crypto-page-index', 'try');
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/crypto/XTVCETH.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
    }elseif($currency == 'xrp'){
      $index = kanews_get_option('crypto-page-index', 'try');
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/crypto/XTVCXRP.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
    }elseif($currency == 'bch'){
      $index = kanews_get_option('crypto-page-index', 'try');
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/crypto/XTVCBCH.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
    }elseif($currency == 'ltc'){
      $index = kanews_get_option('crypto-page-index', 'try');
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/crypto/XTVCLTC.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
    }elseif($currency == 'bnb'){
      $index = kanews_get_option('crypto-page-index', 'try');
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/crypto/XTVCBNB.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
    }elseif($currency == 'sol'){
      $index = kanews_get_option('crypto-page-index', 'try');
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/crypto/XTVCSOL.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
    }elseif($currency == 'avax'){
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/crypto/XTVCAVAX.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
      $index = kanews_get_option('crypto-page-index', 'try');
    }elseif($currency == 'ada'){
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/crypto/XTVCADA.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
      $index = kanews_get_option('crypto-page-index', 'try');
    }elseif($currency == 'dot'){
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/crypto/XTVCDOT.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
      $index = kanews_get_option('crypto-page-index', 'try');
    }elseif($currency == 'doge'){
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/crypto/XTVCDOGE.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
      $index = kanews_get_option('crypto-page-index', 'try');
    }elseif($currency == 'shib'){
      $img2 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/crypto/XTVCSHIB.svg', array('alt' => $currency, 'width' => 18, 'height' => 18), '1e1');
      $img1 = kanews_get_local_image('https://s3-symbol-logo.tradingview.com/country/TR.svg', array('alt' => 'tr', 'width' => 18, 'height' => 18), '1e1');
      $index = kanews_get_option('crypto-page-index', 'try');
    }
    $out .= '<div class="kanews-inline-card-item no-gutters">';
    $out .= '<div class="kanews-inline-card-icon col-auto">';
        $out .= $img1;
        $out .= $img2;
      $out .= '</div>';
      $out .= '<div class="kanews-inline-card-summary col">';
      $percent = '';
      if($data->$currency->change_percent): $percent =  $data->$currency->change_percent.'%'; endif;
      $out .= '<div class="kanews-inline-card-title d-flex justify-content-between"><div>'.$data->$currency->name.'</div><div>'.$percent.'</div></div>';
        $out .= '<div class="kanews-inline-card-content '.$class.'">'.$icon.' '.number_format((float)$data->$currency->value, 2,",",".").' '.strtoupper($index).'</div>';
        $out .= '</div>';
      $out .= '</div>';
  }
  $out .= '</div></div>';

	return $out;
}

/* ------------------------------------------------------------------------- *
	* 14.0 Exhchanges
/* ------------------------------------------------------------------------- */
function kanews_shortcode_exchange( $atts, $content = null ) {
  $atts = shortcode_atts( array(
    "hisse" => '',
    "class"   => ''
  ), $atts );

  $response = get_local_file_contents(KAN_TEMPLATE_PATH."/inc/api/exchange/exchange.json");
  $data = json_decode($response);
  $out = '';

  if($atts['hisse'] != '') {
    foreach ($data as $key => $value) {
      if($key == $atts['hisse']) {
        $out .= '<table class="'.$atts['class'].'" style="table-layout: auto;">';
          $out .= '<thead>';
            $out .= '<tr>';
              $out .= '<th><span class="nobr">'.__( 'Menkul', 'kanews' ).'</span></th>';
              $out .= '<th><span class="nobr">'.__( 'Alış', 'kanews' ).'</span></th>';
              $out .= '<th><span class="nobr">'.__( 'Satış', 'kanews' ).'</span></th>';
              $out .= '<th><span class="nobr">'.__( 'Değişim', 'kanews' ).'</span></th>';
            $out .= '</tr>';
          $out .= '</thead>';
          $out .= ' <tbody>';

            $out .= '<tr>';
            $out .= '<td style="font-weight: 700">'.$key.'</td>';
              if(isset($value->alis)) $out .= '<td>'.$value->alis.'</td>';
              if(isset($value->satis)) $out .= '<td>'.$value->satis.'</td>';
              if(isset($value->degisim)) {
                if ($value->degisim < 0) {
                  $class = 'color-danger';
                  $icon = '<i class="icon-caret icon-180deg icon-2x"></i>';
                } else {
                  $class = 'color-success';
                  $icon = '<i class="icon-caret icon-2x"></i>';
                }
                $out .= '<td class="'.$class.'">'.$icon.' %'.$value->degisim.'</td>';
              }
              $out .= '</tr>'; 
            
          $out .= '</tbody>';
        $out .= '</table>';
        break;
      }
    }
    
  } else {
    $out .= '<table class="'.$atts['class'].'" style="table-layout: auto;">';
      $out .= '<thead>';
        $out .= '<tr>';
          $out .= '<th><span class="nobr">'.__( 'Menkul', 'kanews' ).'</span></th>';
          $out .= '<th><span class="nobr">'.__( 'Alış', 'kanews' ).'</span></th>';
          $out .= '<th><span class="nobr">'.__( 'Satış', 'kanews' ).'</span></th>';
          $out .= '<th><span class="nobr">'.__( 'Değişim', 'kanews' ).'</span></th>';
        $out .= '</tr>';
      $out .= '</thead>';
      $out .= ' <tbody>';
        if(!empty($data)) {
          foreach($data as $key=>$value) {
            $out .= '<tr>';
            $out .= '<td style="font-weight: 700">'.$key.'</td>';
              if(isset($value->alis)) $out .= '<td>'.$value->alis.'</td>';
              if(isset($value->satis)) $out .= '<td>'.$value->satis.'</td>';
              if(isset($value->degisim)) {
                if ($value->degisim < 0) {
                  $class = 'color-danger';
                  $icon = '<i class="icon-caret icon-180deg icon-2x"></i>';
                } else {
                  $class = 'color-success';
                  $icon = '<i class="icon-caret icon-2x"></i>';
                }
                $out .= '<td class="'.$class.'">'.$icon.' %'.$value->degisim.'</td>';
              }
              $out .= '</tr>';

          }
        } else {
          $out .= '<tr><td>'.esc_html__( 'Veri alınamadı!', 'kanews' ).'</td></tr>';
        }
      $out .= '</tbody>';
    $out .= '</table>';
  } 

	return $out;
}

/* ------------------------------------------------------------------------- *
	* 15.0 Sources
/* ------------------------------------------------------------------------- */
function kanews_shortcode_sources( $atts, $content = null ) {
	return sprintf(
		'<div class="%s">%s</div>',
		esc_attr( trim( 'kanews-post-sources') ),
		do_shortcode( $content )
	);
}

/* ------------------------------------------------------------------------- *
	* 16.0 Source
/* ------------------------------------------------------------------------- */
function kanews_shortcode_source( $atts, $content = null ) {
	$atts = shortcode_atts( array(
		"link"  => false,
	), $atts );
	
	$class = 'kanews-post-source d-flex justify-content-between align-items-center';
	$link = isset($atts['link']) && !empty($atts['link']) ? $atts['link'] : '#';

	return sprintf(
		'<div class="%s"><a target="_blank" rel="nofollow" href="%s"><span class="icon-logout"></span> %s</a></div>',
		esc_attr( trim( $class ) ),
		$link,
		do_shortcode( $content )
	);
}

/* ------------------------------------------------------------------------- *
	* 16.0 Date Today
/* ------------------------------------------------------------------------- */
function kan_date_today($atts, $content = null) {
  extract( shortcode_atts( array(
    'format' => ''
  ), $atts ) ); 
  $format = $format ? $format : get_option('date_format');
  return date_i18n( $format, current_time( 'timestamp' ) );
} 
add_shortcode('date-today','kan_date_today'); 

/* ------------------------------------------------------------------------- *
	* 16.0 Date Today
/* ------------------------------------------------------------------------- */
function kanews_shortcode_timeline($atts, $content = null) {
	$atts = shortcode_atts( array(
		"date"  => '10:00',
	), $atts );

  $unique_id = substr(md5($content), 0, 8);

	return sprintf(
		'<div id="timeline-%s" class="%s"><div class="kanews-timeline-news-date"><svg width="18" height="18" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path fill="currentColor" d="M15.09814,12.63379,13,11.42285V7a1,1,0,0,0-2,0v5a.99985.99985,0,0,0,.5.86621l2.59814,1.5a1.00016,1.00016,0,1,0,1-1.73242ZM12,2A10,10,0,1,0,22,12,10.01114,10.01114,0,0,0,12,2Zm0,18a8,8,0,1,1,8-8A8.00917,8.00917,0,0,1,12,20Z"></path></svg> %s</div> <div class="kanews-timeline-news-content">%s %s</div></div>',
    esc_attr($unique_id),
    esc_attr( 'kanews-timeline-news' ),
		$atts['date'],
		do_shortcode( $content ),
    '<div class="timeline-news-footer"><div class="social-share"><a target="_blank" href="//www.twitter.com/intent/tweet?text='.strip_tags($content).' '.get_the_permalink().'#timeline-'.$unique_id.'" class="twitter"><i class="icon-twitter"></i></a> <a target="_blank" href=""//www.facebook.com/sharer/sharer.php?u'.strip_tags($content).' '.get_the_permalink().'#timeline-'.$unique_id.'" class="twitter"><i class="icon-facebook"></i></a></div></div>'
	);
} 

function render_horoscope_calculator() {
    $horoscopes = array(
        'koc' => array(
            'name' => 'Koç', 
            'date' => '21 Mart - 20 Nisan',
            'icon' => 'https://cdn-icons-png.flaticon.com/128/1829/1829158.png',
            'gunluk' => 'Bugün enerjiniz yüksek ve yeni başlangıçlar için ideal bir gün. İş hayatınızda cesur adımlar atabilirsiniz.',
            'haftalik' => 'Bu hafta iletişim kanallarınız açık. Yeni bağlantılar kurabilir ve fikirlerinizi paylaşabilirsiniz.',
            'aylik' => 'Bu ay kariyer odaklı çalışmalarınız meyvesini verebilir. Finansal konularda dikkatli olun.',
            'yillik' => 'Bu yıl kişisel gelişim açısından önemli. Eğitim ve seyahat planlarınızı hayata geçirebilirsiniz.'
        ),
        'boga' => array(
            'name' => 'Boğa', 
            'date' => '21 Nisan - 20 Mayıs',
            'icon' => 'https://cdn-icons-png.flaticon.com/128/1829/1829167.png',
            'gunluk' => 'Bugün pratik çözümler üretme konusunda başarılısınız. Maddi konularda olumlu gelişmeler yaşanabilir.',
            'haftalik' => 'Bu hafta sabit duruşunuz sayesinde zorluklarla başa çıkabilirsiniz. İlişkilerinize özen gösterin.',
            'aylik' => 'Bu ay güvenlik ve istikrar arayışınız ön planda. Yatırımlarınızı gözden geçirmenin tam zamanı.',
            'yillik' => 'Bu yıl sabır ve kararlılık gerektiren projeleriniz için ideal. Sağlam temeller atabilirsiniz.'
        ),
        'ikizler' => array(
            'name' => 'İkizler', 
            'date' => '21 Mayıs - 21 Haziran',
            'icon' => 'https://cdn-icons-png.flaticon.com/128/1829/1829168.png',
            'gunluk' => 'Bugün iletişim yetenekleriniz sayesinde ikna edici olabilirsiniz. Sosyal aktiviteler için ideal bir gün.',
            'haftalik' => 'Bu hafta zihinsel aktiviteleriniz yoğun. Yeni bilgiler edinebilir ve bunları paylaşabilirsiniz.',
            'aylik' => 'Bu ay çeşitlilik arayışınız ön planda. Farklı projeler üzerinde çalışabilir ve esnekliğinizi koruyabilirsiniz.',
            'yillik' => 'Bu yıl iletişim becerilerinizi geliştirmek için ideal. Yayıncılık veya eğitim alanında fırsatlar yakalayabilirsiniz.'
        ),
        'yengec' => array(
            'name' => 'Yengeç', 
            'date' => '22 Haziran - 22 Temmuz',
            'icon' => 'https://cdn-icons-png.flaticon.com/128/1829/1829169.png',
            'gunluk' => 'Bugün duygusal zekanız yüksek. Aile ve ev konularında olumlu gelişmeler yaşanabilir.',
            'haftalik' => 'Bu hafta koruyucu içgüdüleriniz aktif. Sevdiklerinize destek olmak için elinizden geleni yapabilirsiniz.',
            'aylik' => 'Bu ay duygusal güvenlik arayışınız ön planda. Geçmişle ilgili konuları çözüme kavuşturabilirsiniz.',
            'yillik' => 'Bu yıl aile bağlarınızı güçlendirmek için ideal. Ev ve aile konularında önemli adımlar atabilirsiniz.'
        ),
        'aslan' => array(
            'name' => 'Aslan', 
            'date' => '23 Temmuz - 22 Ağustos',
            'icon' => 'https://cdn-icons-png.flaticon.com/128/1829/1829227.png',
            'gunluk' => 'Bugün yaratıcılığınız ve özgüveniniz yüksek. Dikkat çekici ve liderlik gerektiren işlerde başarılı olabilirsiniz.',
            'haftalik' => 'Bu hafta kendinizi ifade etme konusunda başarılısınız. Sanatsal aktivitelere yönelebilirsiniz.',
            'aylik' => 'Bu ay tanınma ve takdir görme arzunuz ön planda. Kariyerinizde öne çıkabilirsiniz.',
            'yillik' => 'Bu yıl kişisel tutkularınızı gerçekleştirmek için ideal. Yaratıcı projeleriniz dikkat çekebilir.'
        ),
        'basak' => array(
            'name' => 'Başak', 
            'date' => '23 Ağustos - 22 Eylül',
            'icon' => 'https://cdn-icons-png.flaticon.com/128/1829/1829228.png',
            'gunluk' => 'Bugün analitik yetenekleriniz sayesinde detaylara dikkat edebilirsiniz. Sağlık konularına özen gösterin.',
            'haftalik' => 'Bu hafta pratik çözümler üretme konusunda başarılısınız. İş hayatınızda verimli olabilirsiniz.',
            'aylik' => 'Bu ay düzen ve organizasyon arayışınız ön planda. Rutinlerinizi iyileştirebilirsiniz.',
            'yillik' => 'Bu yıl becerilerinizi geliştirmek için ideal. İş ve hizmet alanında önemli adımlar atabilirsiniz.'
        ),
        'terazi' => array(
            'name' => 'Terazi', 
            'date' => '23 Eylül - 22 Ekim',
            'icon' => 'https://cdn-icons-png.flaticon.com/128/1829/1829170.png',
            'gunluk' => 'Bugün diplomatik yetenekleriniz sayesinde ilişkilerinizde denge sağlayabilirsiniz. Estetik konulara ilgi duyabilirsiniz.',
            'haftalik' => 'Bu hafta ortaklık ve işbirliği konularında başarılısınız. İlişkilerinizi güçlendirebilirsiniz.',
            'aylik' => 'Bu ay adalet ve denge arayışınız ön planda. Hukuki konularda olumlu gelişmeler yaşanabilir.',
            'yillik' => 'Bu yıl ilişkilerinizi geliştirmek için ideal. Evlilik veya ortaklık konularında önemli adımlar atabilirsiniz.'
        ),
        'akrep' => array(
            'name' => 'Akrep', 
            'date' => '23 Ekim - 21 Kasım',
            'icon' => 'https://cdn-icons-png.flaticon.com/128/1829/1829171.png',
            'gunluk' => 'Bugün sezgileriniz güçlü. Derin araştırmalar yapabilir ve gizli kalmış konuları keşfedebilirsiniz.',
            'haftalik' => 'Bu hafta tutkulu ve kararlısınız. Hedeflerinize ulaşmak için gerekli adımları atabilirsiniz.',
            'aylik' => 'Bu ay dönüşüm ve yenilenme arayışınız ön planda. Kendinizi yeniden keşfedebilirsiniz.',
            'yillik' => 'Bu yıl içsel gücünüzü keşfetmek için ideal. Finansal konularda önemli adımlar atabilirsiniz.'
        ),
        'yay' => array(
            'name' => 'Yay', 
            'date' => '22 Kasım - 21 Aralık',
            'icon' => 'https://cdn-icons-png.flaticon.com/128/1829/1829229.png',
            'gunluk' => 'Bugün iyimserliğiniz ve macera arzunuz yüksek. Yeni deneyimler için ideal bir gün.',
            'haftalik' => 'Bu hafta geniş vizyonunuz sayesinde fırsatları görebilirsiniz. Eğitim konularında başarılı olabilirsiniz.',
            'aylik' => 'Bu ay özgürlük ve genişleme arayışınız ön planda. Seyahat planları yapabilirsiniz.',
            'yillik' => 'Bu yıl ufkunuzu genişletmek için ideal. Yüksek öğrenim veya yurtdışı bağlantılı konularda ilerleyebilirsiniz.'
        ),
        'oglak' => array(
            'name' => 'Oğlak', 
            'date' => '22 Aralık - 20 Ocak',
            'icon' => 'https://cdn-icons-png.flaticon.com/128/1829/1829230.png',
            'gunluk' => 'Bugün disiplinli ve kararlısınız. Kariyer hedeflerinize ulaşmak için gerekli adımları atabilirsiniz.',
            'haftalik' => 'Bu hafta sorumluluk bilinci yüksek. İş hayatınızda başarılı olabilirsiniz.',
            'aylik' => 'Bu ay statü ve başarı arayışınız ön planda. Hedeflerinize ulaşmak için stratejik davranabilirsiniz.',
            'yillik' => 'Bu yıl kariyer hedeflerinizi gerçekleştirmek için ideal. Uzun vadeli planlarınızı hayata geçirebilirsiniz.'
        ),
        'kova' => array(
            'name' => 'Kova', 
            'date' => '21 Ocak - 18 Şubat',
            'icon' => 'https://cdn-icons-png.flaticon.com/128/1829/1829231.png',
            'gunluk' => 'Bugün yenilikçi fikirleriniz sayesinde farklı çözümler üretebilirsiniz. Teknoloji konularında başarılı olabilirsiniz.',
            'haftalik' => 'Bu hafta özgün düşünce tarzınız dikkat çekici. Grup çalışmalarında öne çıkabilirsiniz.',
            'aylik' => 'Bu ay özgürlük ve bireysellik arayışınız ön planda. Sosyal çevrenizi genişletebilirsiniz.',
            'yillik' => 'Bu yıl ideallerinizi gerçekleştirmek için ideal. İnsanlık yararına projeler geliştirebilirsiniz.'
        ),
        'balik' => array(
            'name' => 'Balık', 
            'date' => '19 Şubat - 20 Mart',
            'icon' => 'https://cdn-icons-png.flaticon.com/128/1829/1829172.png',
            'gunluk' => 'Bugün sezgileriniz ve hayal gücünüz yüksek. Sanatsal aktiviteler için ideal bir gün.',
            'haftalik' => 'Bu hafta empatik yaklaşımınız sayesinde insanlarla bağ kurabilirsiniz. Manevi konulara ilgi duyabilirsiniz.',
            'aylik' => 'Bu ay içsel huzur arayışınız ön planda. Meditasyon ve dinlenmeye zaman ayırabilirsiniz.',
            'yillik' => 'Bu yıl ruhsal gelişiminiz için ideal. Yaratıcı potansiyelinizi keşfedebilirsiniz.'
        )
    );

    ob_start(); ?>
    <form id="burc_calc" class="kanews-calculator">
        <div id="horoscope-data" style="display:none;" 
            <?php foreach($horoscopes as $sign => $data): ?>
                data-<?php echo esc_attr($sign); ?>='<?php echo json_encode($data); ?>'
            <?php endforeach; ?>>
        </div>
        <div class="row">
            <div class="col-12">
                <div class="kanews-form">
                    <select id="horoscope_sign" aria-label="<?php esc_html_e('Burç Seçimi', 'kanews'); ?>">
                        <option value=""><?php _e('Burç Seçiniz', 'kanews'); ?></option>
                        <?php foreach($horoscopes as $key => $horoscope): ?>
                            <option value="<?php echo esc_attr($key); ?>">
                                <?php echo esc_html($horoscope['name']); ?> (<?php echo esc_html($horoscope['date']); ?>)
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <label><?php _e('Burç Seçimi', 'kanews'); ?></label>
                </div>
            </div>
            <div class="col-12">
                <div class="kanews-form">
                    <select id="horoscope_period" aria-label="<?php esc_html_e('Dönem', 'kanews'); ?>">
                        <option value="gunluk"><?php _e('Günlük', 'kanews'); ?></option>
                        <option value="haftalik"><?php _e('Haftalık', 'kanews'); ?></option>
                        <option value="aylik"><?php _e('Aylık', 'kanews'); ?></option>
                        <option value="yillik"><?php _e('Yıllık', 'kanews'); ?></option>
                    </select>
                    <label><?php _e('Dönem', 'kanews'); ?></label>
                </div>
            </div>
            <div class="col-12">
                <button type="button" id="horoscope_calculate" class="kanews-btn kanews-btn-sm mt-0 mb-0 kanews-btn-block"><?php _e('Yorumları Göster', 'kanews'); ?></button>
            </div>
            <div class="col-12 mt-1">
                <div id="horoscope_result" class="kanews-form p-3" style="display: none;">
                    <div class="horoscope-header d-flex align-items-center">
                        <div class="horoscope-icon mr-2">
                            <img id="horoscope_icon" src="" alt="" width="64" height="64">
                        </div>
                        <div class="horoscope-title">
                            <h3 id="horoscope_title"></h3>
                            <p id="horoscope_date" class="text-muted mb-0"></p>
                        </div>
                    </div>
                    <div class="horoscope-content mt-3">
                        <p id="horoscope_text"></p>
                    </div>
                </div>
            </div>
        </div>
        <div class="calculator-info">
            <div class="info-row">
                <span class="info-label"><?php _e('Seçilen Burç', 'kanews'); ?></span>
                <span class="info-value selected-value">-</span>
            </div>
            <div class="info-row">
                <span class="info-label"><?php _e('Dönem', 'kanews'); ?></span>
                <span class="info-value conversion-info">-</span>
            </div>
        </div>
    </form>
    <?php
    return ob_get_clean();
} 

function render_credit_calculator() {
    ob_start(); ?>
    <form id="kredi_calc" class="kanews-calculator">
        <div class="row">
            <div class="col-12">
                <div class="kanews-form">
                    <input type="text" id="kredi_amount" placeholder="<?php esc_html_e('Kredi tutarını giriniz', 'kanews'); ?>">
                    <label><?php _e('Kredi Tutarı (₺)', 'kanews'); ?></label>
                </div>
            </div>
            <div class="col-12">
                <div class="kanews-form">
                    <input type="text" id="kredi_interest" placeholder="<?php esc_html_e('Faiz oranını giriniz', 'kanews'); ?>">
                    <label><?php _e('Yıllık Faiz Oranı (%)', 'kanews'); ?></label>
                </div>
            </div>
            <div class="col-12">
                <div class="kanews-form">
                    <select id="kredi_term">
                        <option value=""><?php _e('Vade Seçiniz', 'kanews'); ?></option>
                        <option value="12"><?php _e('12 Ay (1 Yıl)', 'kanews'); ?></option>
                        <option value="24"><?php _e('24 Ay (2 Yıl)', 'kanews'); ?></option>
                        <option value="36"><?php _e('36 Ay (3 Yıl)', 'kanews'); ?></option>
                        <option value="48"><?php _e('48 Ay (4 Yıl)', 'kanews'); ?></option>
                        <option value="60"><?php _e('60 Ay (5 Yıl)', 'kanews'); ?></option>
                        <option value="120"><?php _e('120 Ay (10 Yıl)', 'kanews'); ?></option>
                    </select>
                    <label><?php _e('Vade', 'kanews'); ?></label>
                </div>
            </div>
            <div class="col-12">
                <div class="kanews-form">
                    <select id="kredi_type">
                        <option value="esit_taksit"><?php _e('Eşit Taksitli', 'kanews'); ?></option>
                        <option value="esit_anapara"><?php _e('Eşit Anaparalı', 'kanews'); ?></option>
                    </select>
                    <label><?php _e('Ödeme Tipi', 'kanews'); ?></label>
                </div>
            </div>
            <div class="col-12">
                <button type="button" id="kredi_calculate" class="kanews-btn kanews-btn-sm mt-0 mb-0 kanews-btn-block"><?php _e('Hesapla', 'kanews'); ?></button>
            </div>
        </div>
        <div class="calculator-info">
            <div class="info-row">
                <span class="info-label"><?php _e('Aylık Taksit', 'kanews'); ?></span>
                <span class="info-value monthly-payment">-</span>
            </div>
            <div class="info-row">
                <span class="info-label"><?php _e('Toplam Ödeme', 'kanews'); ?></span>
                <span class="info-value total-payment">-</span>
            </div>
            <div class="info-row">
                <span class="info-label"><?php _e('Toplam Faiz', 'kanews'); ?></span>
                <span class="info-value total-interest">-</span>
            </div>
        </div>
    </form>
    <?php
    return ob_get_clean();
} 


function render_savings_calculator() {
    ob_start();
    ?>
    <div class="kanews-calculator" id="birikim_calc">
        <form class="kanews-form" action="#" method="post">
            <div class="kanews-form">
                <input type="text" id="savings_initial" name="savings_initial" class="kanews-form-control" placeholder="<?php _e('Başlangıç ​​tutarını girin', 'kanews'); ?>" required>
                <label for="savings_initial"><?php _e('İlk Mevduat (₺)', 'kanews'); ?></label>

            </div>
            <div class="kanews-form">
                <input type="text" id="savings_monthly" name="savings_monthly" class="kanews-form-control" placeholder="<?php _e('Aylık tutarı girin', 'kanews'); ?>" required>
                <label for="savings_monthly"><?php _e('Aylık Yatırım (₺)', 'kanews'); ?></label>
            </div>
            <div class="row">
                <div class="col-6">
                    <div class="kanews-form">
                        <input type="text" id="savings_interest" name="savings_interest" class="kanews-form-control" placeholder="<?php _e('Faiz oranını girin', 'kanews'); ?>" required>
                        <label for="savings_interest"><?php _e('Faiz Oranı (%)', 'kanews'); ?></label>
                    </div>
                </div>
                <div class="col-6">
                    <div class="kanews-form">
                        <input type="number" id="savings_years" name="savings_years" class="kanews-form-control" min="1" max="50" value="5" required>
                        <label for="savings_years"><?php _e('Yıl', 'kanews'); ?></label>
                    </div>
                </div>
            </div>
            <div class="kanews-form-group">
                <button type="button" id="savings_calculate" class="kanews-btn kanews-btn-primary"><?php _e('Hesapla', 'kanews'); ?></button>
            </div>
            
            
            
            <div class="calculator-info">
                <div class="info-row">
                    <div class="info-label"><?php _e('Faiz Yıllık', 'kanews'); ?></div>
                    <div class="info-value"><?php _e('Aylık', 'kanews'); ?></div>
                </div>
                <div class="savings-breakdown" style="display: none;">
                <div class="savings-item total-savings-item info-row">
                    <div class="savings-label info-label"><?php _e('Toplam Yatırım', 'kanews'); ?></div>
                    <div class="savings-value info-value" id="total_savings_value">₺0</div>
                </div>
                <div class="savings-item total-contribution-item info-row">
                    <div class="savings-label info-label"><?php _e('Toplam Yatırım', 'kanews'); ?></div>
                    <div class="savings-value info-value" id="total_contribution_value">₺0</div>
                </div>
                <div class="savings-item total-interest-item info-row">
                    <div class="savings-label info-label"><?php _e('Toplam Faiz', 'kanews'); ?></div>
                    <div class="savings-value info-value" id="total_interest_value">₺0</div>
                </div>
            </div>
            </div>
        </form>
    </div>
    <?php
    return ob_get_clean();
}
